package de.renew.simulator.api;


import org.apache.log4j.Logger;

import de.renew.plugin.ServiceLookupException;
import de.renew.plugin.ServiceLookupInfrastructure;
import de.renew.simulatorontology.simulation.SimulatorExtension;

/**
 * Adapter for {@code SimulatorExtensions}. Uses the adapter pattern to delegate
 * calls to the implementation
 */
public final class SimulatorExtensions {
    private static final ISimulatorExtensions SIMULATOR_EXTENSIONS;
    private static final Logger LOGGER = Logger.getLogger(SimulatorExtensions.class);
    static {
        try {
            SIMULATOR_EXTENSIONS = ServiceLookupInfrastructure.getInstance()
                .getFirstServiceProvider(ISimulatorExtensions.class);
        } catch (ServiceLookupException e) {
            LOGGER.error(
                "Could not find a service provider for " + ISimulatorExtensions.class + ": ", e);
            throw new RuntimeException(e);
        }
    }

    private SimulatorExtensions() {}

    /**
     * Registers an extension to the simulation component. The extension will
     * become active the next time a new simulation environment is set up.
     * Active extensions will be notified about several events (see
     * {@link SimulatorExtension} interface).
     *
     * @param extension the extension to register. Duplicate registrations will be
     *        ignored.
     */
    public static void addExtension(SimulatorExtension extension) {
        SIMULATOR_EXTENSIONS.addExtension(extension);
    }

    /**
     * Removes and deregisters the given {@code SimulatorExtension} from the simulation engine.
     * The deregistration does not cancel active extensions from any currently running simulation,
     * but they will not be activated again when the next simulation is set up.
     *
     * @param extension the extension to deregister. If the extension was not
     *        registered before, this method call is ignored.s
     */
    public static void removeExtension(SimulatorExtension extension) {
        SIMULATOR_EXTENSIONS.removeExtension(extension);
    }
}