package de.renew.simulator.api;

import org.apache.log4j.Logger;

import de.renew.plugin.ServiceLookupException;
import de.renew.plugin.ServiceLookupInfrastructure;
import de.renew.simulatorontology.loading.Finder;
import de.renew.simulatorontology.loading.PathlessFinder;

/**
 * This class acts as the main facade for adding and removing {@link Finder} and {@link PathlessFinder}
 * to the simulator.
 */
public final class FinderRegistration {

    private static final Logger LOGGER = Logger.getLogger(FinderRegistration.class);
    private static final IFinderRegistration REGISTRATION;

    static {
        try {
            REGISTRATION = ServiceLookupInfrastructure.getInstance()
                .getFirstServiceProvider(IFinderRegistration.class);
        } catch (ServiceLookupException e) {
            LOGGER.error(
                "Could not find a service provider for " + IFinderRegistration.class + ": ", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * Private constructor, as no instances of this class should exist.
     */
    private FinderRegistration() {}

    /**
     * Registers a finder for the default shadow net loader. Finders are without
     * effect if the default shadow net loader is overridden (either as plug-in
     * configuration or by an individual shadow net system).
     *
     * @param finder the shadow net file finder to add to the default shadow net
     *        loader.
     */
    public static void registerDefaultNetFinder(Finder finder) {
        REGISTRATION.registerDefaultNetFinder(finder);
    }

    /**
     * Deregisters a finder from the default shadow net loader.
     *
     * @param finder the shadow net file finder to remove from the default shadow
     *        net loader.
     */
    public static void removeDefaultNetFinder(Finder finder) {
        REGISTRATION.removeDefaultNetFinder(finder);
    }

    /**
     * Registers a pathless finder for the default shadow net loader. Finders
     * are without effect if the default shadow net loader is overridden (either
     * as plug-in configuration or by an individual shadow net system).
     *
     * @param finder the shadow net file finder to add to the default shadow net
     *        loader.
     */
    public static void registerDefaultPathlessFinder(PathlessFinder finder) {
        REGISTRATION.registerDefaultPathlessFinder(finder);
    }

    /**
     * Deregisters a pathless finder from the default shadow net loader.
     *
     * @param finder the shadow net file finder to remove from the default shadow
     *        net loader.
     */
    public static void removeDefaultPathlessFinder(final PathlessFinder finder) {
        REGISTRATION.removeDefaultPathlessFinder(finder);
    }
}
