package de.renew.remote;

import java.rmi.Remote;
import java.rmi.RemoteException;


/**
 * Represents the marking of a place instance as a snapshot.
 * Its contents will not change even if transitions fire.
 *
 * <p>
 * This is a remote interface that allows an implementation
 * that can be used via RMI. This accessor does not extend the
 * <code>de.renew.remote.Accessor</code> interface because it
 * does not represent an object that locally exists.
 * </p>
 *
 * @author Olaf Kummer
 */
public interface MarkingAccessor extends Remote {

    /**
     * Return the number of distinct token values in this marking.
     * As long as two objects in the place marking are equal,
     * they are reported as one distinct token.
     *
     * @return the number of distinct tokens
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public int getDistinctTokenCount() throws RemoteException;

    /**
     * Return the number of free tokens in this marking.
     * Even if two tokens are equal, they are both counted individually.
     * Currently tested tokens are not included in this count.
     *
     * @return the number of free tokens
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public int getFreeTokenCount() throws RemoteException;

    /**
     * Return the number of currently tested tokens in this marking.
     *
     * @return the number of tested tokens
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public int getTestedTokenCount() throws RemoteException;

    /**
     * Return the string representation of each object in the set of
     * tokens. The strings are generated by the <code>toString()</code>
     * methods.
     *
     * This method produces the same result as like a call to
     * <code>getTokenString</code> for each token position
     * and putting the results into an array would do.
     *
     * @return string representations for each object
     * @exception java.rmi.RemoteException if an RMI failure occured.
     *
     * @see #getTokenString(int)
     */
    public String[] getAllTokenStrings() throws RemoteException;

    /**
     * Return the numbers of free tokens that equal each object
     * in the set of tokens.
     *
     * This method produces the same result as like a call to
     * <code>getTokenFreeCount</code> for each token position
     * and putting the results into an array would do.
     *
     * @return free token counts for each object
     * @exception java.rmi.RemoteException if an RMI failure occured.
     *
     * @see #getTokenFreeCount(int)
     */
    public int[] getAllTokenFreeCounts() throws RemoteException;

    /**
     * Return whether the objects in the set of tokens are
     * currently being tested.
     *
     * This method produces the same result as like a call to
     * <code>getTokenTested</code> for each token position
     * and putting the results into an array would do.
     *
     * @return array of booleans (see <code>getTokenTested</code>
     *                            for an explanation)
     * @exception java.rmi.RemoteException if an RMI failure occured.
     *
     * @see #getTokenTested(int)
     */
    public boolean[] getAllTokenTested() throws RemoteException;

    /**
     * Return the number of free tokens that equal the object at the
     * given index position.
     *
     * @param i the index of the token that has to be counted,
     *          must be between 0 and getDistinctTokenCount()-1.
     * @return the number of free tokens that equal the token
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public int getTokenFreeCount(int i) throws RemoteException;

    /**
     * Return whether the object at the given index position is currently being tested.
     *
     * @param i the index of the token that might be tested,
     *          must be between 0 and getDistinctTokenCount()-1.
     * @return <code>true</code>,  if the token is being tested
     *       - <code>false</code>, otherwise
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public boolean getTokenTested(int i) throws RemoteException;

    /**
     * Return the string representation of an object at a given
     * index position. The string is generated by the toString() method.
     *
     * @param i the index of the token to describe,
     *          must be between 0 and getDistinctTokenCount()-1.
     * @return the string representation
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public String getTokenString(int i) throws RemoteException;

    /**
     * Returns the free time array an object at a given index position.
     *
     * @param i the index of the token to describe,
     *          must be between 0 and getDistinctTokenCount()-1.
     * @return the free time array.
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public double[] getTokenTimes(int i) throws RemoteException;

    /**
     * Returns the free time multiplicity array an object at a given index position.
     *
     * @param i the index of the token to describe,
     *          must be between 0 and getDistinctTokenCount()-1.
     * @return the free time multiplicity array.
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public int[] getTokenTimeMultiplicities(int i) throws RemoteException;

    /**
     * Return an object that describes the token at a given
     * index position.
     *
     * @param i the index of the token to return,
     *          must be between 0 and getDistinctTokenCount()-1.
     * @return the object accessor
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public ObjectAccessor getToken(int i) throws RemoteException;

    /**
     * Returns the current time that was used for the time set.
     * @return The former current time.
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public double getCurrentTime() throws RemoteException;

    /**
     * Try to remove one of the tokens at a given index.
     * If concurrent actions have already removed the token, return false.
     * The given index must be between 0 and getDistinctTokenCount()-1.
     *
     * <p>
     * To add a token to a place, use the <code>addSerializableToken</code>
     * method of the place accessor from which this marking accessor
     * was obtained. The change will not be reflected in this marking
     * accessor instance, naturally.
     * </p>
     *
     * @see PlaceInstanceAccessor#addSerializableToken
     *
     * @param i the index of the token to be removed
     * @return true, if the token was successfully removed
     * @exception java.rmi.RemoteException if an RMI failure occured.
     */
    public boolean removeOneOf(int i) throws RemoteException;
}