package de.renew.propertymanagement.prop;

import java.util.EventObject;


/**
 * The PropertyChangeEvent class represents an event that indicates a change in a configurable property.
 * It extends the standard Java EventObject class and provides additional information about the property change,
 * including the property name, the old value, and the new value.
 */
public class PropertyChangeEvent extends EventObject {

    /**
     * The name of the configurable property that has changed.
     */
    private final String _property;

    /**
     * The old value of the configurable property before the change.
     */
    private final String _oldValue;

    /**
     * The new value of the configurable property after the change.
     */
    private final String _newValue;

    /**
     * Constructs a PropertyChangeEvent with the specified source, property name, old value, and new value.
     *
     * @param source The object on which the event initially occurred.
     * @param property The name of the configurable property that has changed.
     * @param oldValue The old value of the configurable property.
     * @param newValue The new value of the configurable property.
     */
    public PropertyChangeEvent(Object source, String property, String oldValue, String newValue) {
        super(source);
        _property = property;
        _oldValue = oldValue;
        _newValue = newValue;
    }

    /**
     * Constructs a PropertyChangeEvent with the specified source, property name, and new value.
     * Assumes that the property is newly added, and there is no old value.
     *
     * @param source The object on which the event initially occurred.
     * @param property The name of the configurable property that has changed.
     * @param newValue The new value of the configurable property.
     */
    public PropertyChangeEvent(Object source, String property, String newValue) {
        this(source, property, null, newValue);
    }

    /**
     * Gets the name of the configurable property that has changed.
     *
     * @return The name of the configurable property.
     */
    public String getProperty() {
        return _property;
    }

    /**
     * Gets the old value of the configurable property before the change.
     *
     * @return The old value of the configurable property. Null if it is not known.
     */
    public String getOldValue() {
        return _oldValue;
    }

    /**
     * Gets the new value of the configurable property after the change.
     *
     * @return The new value of the configurable property. Null if there is no new value (e.g. property is removed).
     */
    public String getNewValue() {
        return _newValue;
    }
}
