package de.renew.propertymanagement.prop;

/**
 * The {@link ConfigurableProperty} class represents a single configurable property with its default and current values,
 * along with additional metadata such as the plugin name, display name, and key.
 * Instances of this class are used within the ConfigurablePropertyManager to manage and track properties.
 */
public class ConfigurableProperty {
    /**
     * The default value of the configurable property.
     */
    private final String _defaultValue;

    /**
     * The current value of the configurable property.
     */
    private String _currentValue;

    /**
     * The name of the plugin associated with the configurable property (nullable).
     */
    private final String _pluginName;

    /**
     * The display name of the configurable property (nullable).
     */
    private final String _displayName;

    /**
     * The key identifying the configurable property.
     */
    private final String _key;

    /**
     * The possible options for the property (nullable).
     */
    private final String[] _selectionChoices;

    /**
     * Constructs a ConfigurableProperty with the specified key, default value, plugin name, and display name.
     *
     * @param key The key identifying the configurable property.
     * @param defaultValue The default value of the configurable property (nullable).
     * @param pluginName The name of the plugin associated with the configurable property (nullable).
     * @param displayName The display name of the configurable property (nullable).
     * @param selectionChoices The possible options for the property (nullable).
     */
    public ConfigurableProperty(
        String key, String defaultValue, String pluginName, String displayName,
        String[] selectionChoices)
    {
        this._defaultValue = defaultValue;
        this._pluginName = pluginName;
        this._displayName = displayName;
        this._key = key;
        this._currentValue = null;
        this._selectionChoices = selectionChoices;
    }

    /**
     * Constructs a ConfigurableProperty with the specified key and current value.
     *
     * @param key The key identifying the configurable property.
     * @param currentValue The current value of the configurable property.
     */
    public ConfigurableProperty(String key, String currentValue) {
        this(key, currentValue, null, null, null);
        setCurrentValue(currentValue);
    }

    /**
     * Sets the current value of the configurable property.
     *
     * @param currentValue The new current value.
     */
    public void setCurrentValue(String currentValue) {
        this._currentValue = currentValue;
    }

    /**
     * Gets the default value of the configurable property.
     *
     * @return The default value.
     */
    public String getDefaultValue() {
        return _defaultValue;
    }

    /**
     * Gets the current value of the configurable property.
     *
     * @return The current value.
     */
    public String getCurrentValue() {
        return _currentValue;
    }

    /**
     * Gets the name of the plugin associated with the configurable property.
     *
     * @return The plugin name (nullable).
     */
    public String getPluginName() {
        return _pluginName;
    }

    /**
     * Gets the display name of the configurable property.
     *
     * @return The display name (nullable).
     */
    public String getDisplayName() {
        return _displayName;
    }

    /**
     * Gets the key identifying the configurable property.
     *
     * @return The key.
     */
    public String getKey() {
        return _key;
    }

    /**
     * Gets the possible options for the property.
     *
     * @return The selection choices (nullable).
     */
    public String[] getSelectionChoices() {
        return _selectionChoices;
    }

    /**
     * Checks if the current value is equal to the default value, indicating whether the property is at its default state.
     *
     * @return True if the current value is null or equal to the default value, false otherwise.
     */
    public boolean isDefault() {
        return _currentValue == null || _currentValue.equals(_defaultValue);
    }
}
