package de.renew.gui.tool;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.util.function.Supplier;

import CH.ifa.draw.standard.CreationTool;
import de.renew.draw.storables.ontology.Figure;
import de.renew.draw.ui.ontology.DrawingEditor;

/**
 * Creation tool that uses a specified default size when creating new figures.
 *
 * This tool extends the standard CreationTool by allowing the specification of a default size
 * for new figures. When a new figure is created and initially empty, it is resized to the
 * default dimensions centered around the initial click location.
 */
public class DefaultSizeCreationTool extends CreationTool {
    private final Supplier<Figure> figureSupplier;
    private final Dimension defaultDimension;

    /**
     * Constructs a DefaultSizeCreationTool with the specified drawing editor, figure supplier, and default dimension.
     *
     * @param editor the drawing editor that manages the drawing
     * @param figureSupplier a supplier that provides instances of the figure to be created
     * @param defaultDimension the default dimension to be used for the created figures when they are initially empty
     */
    public DefaultSizeCreationTool(
        DrawingEditor editor, Supplier<Figure> figureSupplier, Dimension defaultDimension)
    {
        super(editor);
        this.figureSupplier = figureSupplier;
        this.defaultDimension = defaultDimension;
    }

    @Override
    protected Figure createFigure() {
        return this.figureSupplier.get();
    }

    @Override
    public void mouseUp(MouseEvent e, int x, int y) {
        Figure created = createdFigure();
        if (created.isEmpty()) {
            Point loc = created.displayBox().getLocation();
            Dimension d = defaultDimension;
            int w2 = d.width / 2;
            int h2 = d.height / 2;
            created.displayBox(
                new Point(loc.x - w2, loc.y - h2),
                new Point(loc.x - w2 + d.width, loc.y - h2 + d.height));
        }
        super.mouseUp(e, x, y);
    }
}
