package de.renew.gui.pnml.creator;

import java.awt.Point;

import org.w3c.dom.Element;

import CH.ifa.draw.figures.TextFigure;
import CH.ifa.draw.framework.ParentFigure;
import de.renew.gui.DeclarationFigure;
import de.renew.gui.pnml.converter.Converter;


/**
 * A XMLCreator that parses Renew annotations.
 * Inside the {@link de.renew.gui.CPNDrawing} of the net the annotation is represented by a {@link TextFigure}.
 * The name of the tag representing the annotation in the PNML file is given to the constructors.
 * The tag is created by the {@link #createElement(PNMLCreator, Object)} method
 * which is given a {@link TextFigure}.
 * <p>
 * The {@link AttributeCreator} creates similar tag structures without requiring a {@link TextFigure}.
 *
 * @author 0schlein
 */
public class AnnotationCreator extends ElementCreator {
    private final boolean parentVirtual;

    /**
     * Create a XMLCreator that parses Renew annotations.
     * @param tagName name to use for saving the parsed annotation
     */
    public AnnotationCreator(String tagName, Converter converter) {
        this(tagName, false, converter);
    }

    public AnnotationCreator(String tagName, boolean parentVirtual, Converter converter) {
        super(tagName, converter);
        this.parentVirtual = parentVirtual;
    }

    /**
     * @return the saved object cast to a renew TextFigure
     */
    protected TextFigure getTextFigure() {
        return (TextFigure) getObject();
    }

    /**
     * Inside the tag (created with the tag name given to the constructors)
     * a {@code text} tag will contain the text of the {@link TextFigure}
     * given to {@link #createElement(PNMLCreator, Object)}
     * and a {@code graphics} tag will contain information to the graphical representation.
     * Annotations that were inscribed to a virtual place will store this
     * information inside a {@code toolspecific} tag.
     */
    @Override
    protected Element doCreateElement(PNMLCreator pnmlCreator) {
        Element element = super.doCreateElement(pnmlCreator);
        Element textEle = pnmlCreator.createTextElement(getTextFigure().getText());
        element.appendChild(createGraphic(pnmlCreator));
        element.appendChild(textEle);
        if (parentVirtual) {
            Element toolSpec = pnmlCreator.createToolspecific();
            Element vir = pnmlCreator.createElement("virtual");
            toolSpec.appendChild(vir);
            element.appendChild(toolSpec);
        }

        return element;
    }

    /**
     * parse the graphical information saved with this annotation and append it to the created xml element
     * @return
     */
    protected Element createGraphic(PNMLCreator pnmlCreator) {
        GraphicCreator graphicCreator = new GraphicCreator(pnmlCreator);
        Point center = getTextFigure().center();
        ParentFigure parent = getTextFigure().parent();
        if (parent == null) {
            if (!(getTextFigure() instanceof DeclarationFigure)) {
                graphicCreator.addOffset(center.x, center.y);
            } else {
                Point origin = getTextFigure().getOrigin();
                graphicCreator.addOffset(origin.x, origin.y);
            }
        } else {
            Point centerParent = parent.center();
            graphicCreator.addOffset(center.x - centerParent.x, center.y - centerParent.y);
        }
        return graphicCreator.getElement();
    }
}