package de.renew.gui.configure;

import java.awt.Component;
import java.util.Properties;

import de.renew.gui.ConfigureSimulationTabController;
import de.renew.plugin.PropertyHelper;
import de.renew.remote.RemoteExtension;
import de.renew.remote.Server;


/**
 * An option panel controller to configure remote access properties.
 * @author Michael Duvigneau
 */
public class ConfigureRemoteAccessController implements ConfigureSimulationTabController {
    private static final org.apache.log4j.Logger logger =
        org.apache.log4j.Logger.getLogger(ConfigureRemoteAccessController.class);
    private final ConfigureRemoteAccessTab tab;

    /**
     * Creates a ConfigureRemoteAccessTab and sets it as the current tab for this controller.
     */
    public ConfigureRemoteAccessController() {
        this.tab = new ConfigureRemoteAccessTab();
        this.tab.addEnabledStateListener(e -> this.enabledStateChanged());
    }

    @Override
    public Component getTab() {
        return tab;
    }

    @Override
    public void commitTab(Properties props) {
        boolean enabled = tab.getRemoteEnabled();
        props.setProperty(RemoteExtension.ENABLE_PROP_NAME, Boolean.toString(enabled));
        String publicName = tab.getPublicName().trim();
        if (publicName.isEmpty() || publicName.equals(Server.DEFAULT_SERVER_NAME)) {
            props.remove(RemoteExtension.NAME_PROP_NAME);
            publicName = null;
        } else {
            props.setProperty(RemoteExtension.NAME_PROP_NAME, publicName);
        }
        String className = tab.getServerClass().trim();
        if (className.isEmpty()
            || className.equals(RemoteExtension.DEFAULT_SERVER_CLASS.getName())) {
            props.remove(RemoteExtension.CLASS_PROP_NAME);
            className = null;
        } else {
            props.setProperty(RemoteExtension.CLASS_PROP_NAME, className);
        }

        logger.debug(
            "ConfigureRemoteAccessController: " + "Configured enabled=" + enabled + ", name="
                + publicName + ", class=" + className + ".");
    }

    @Override
    public void updateTab(Properties props) {
        tab.setRemoteEnabled(
            PropertyHelper.getBoolProperty(props, RemoteExtension.ENABLE_PROP_NAME));
        String className = props.getProperty(RemoteExtension.CLASS_PROP_NAME);
        if (className == null) {
            className = RemoteExtension.DEFAULT_SERVER_CLASS.getName();
        }
        tab.setServerClass(className);
        String publicName = props.getProperty(RemoteExtension.NAME_PROP_NAME);
        if (publicName == null) {
            publicName = Server.DEFAULT_SERVER_NAME;
        }
        tab.setPublicName(publicName);
        String factoryName = props.getProperty("de.renew.remote.socketFactory");
        if (factoryName == null) {
            factoryName = "";
        }
        tab.setSocketFactory(factoryName);
        enabledStateChanged();
    }


    /**
     * This method is called by the associated Swing component
     * whenever the user changes the enabledness of the remote
     * access.
     * It instructs the dialog tab to en- or disable the detailed
     * options group.
     **/
    public void enabledStateChanged() {
        tab.enableDetailOptions(tab.getRemoteEnabled());
    }
}