/**
 * Package containing feature structure and type system implementations.
 */
package de.uni_hamburg.fs;

/**
 * Defines operations for handling uncompiled types during parsing phase.
 * Parsed types serve as intermediate representations that can be unified
 * with other types and eventually compiled into concrete type instances.
 *
 * @see java.io.Serializable
 */
public interface ParsedType extends java.io.Serializable {

    /**
     * The most general parsed type in the hierarchy.
     * Represents a parsed type with an empty concept set that acts as
     * the identity element for unification operations. When any parsed type
     * is united with PARSED_TOP, it remains unchanged.
     */
    public final static ParsedType PARSED_TOP = new ParsedConjunctiveType(new ConceptSet());

    /**
     * Combines this parsed type with another parsed type.
     * This operation is a preliminary form of unification between two parsed types.
     *
     * @param that the parsed type to unite with this one
     * @return the resulting unified parsed type
     * @throws UnificationFailure if the types cannot be united
     */
    public ParsedType unite(ParsedType that) throws UnificationFailure;

    /**
     * Converts this parsed type into a compiled "real" type.
     * This method performs the actual compilation of the parsed type structure.
     *
     * @return the compiled type
     * @throws UnificationFailure if the type cannot be compiled
     */
    public Type asType() throws UnificationFailure;
}