package de.renew.export.io.exportFormats;

import java.net.URL;
import java.util.Iterator;

import org.freehep.graphicsio.PageConstants;

import de.renew.draw.storables.api.StorableApi;
import de.renew.plugin.IPlugin;
import de.renew.plugin.PluginAdapter;
import de.renew.plugin.PluginException;
import de.renew.plugin.PluginManager;
import de.renew.plugin.PluginProperties;


/**
 * @author Benjamin Schleinzer
 *
 */
public class ExportPlugin extends PluginAdapter {
    /**
     * The Logger
     */
    public static final org.apache.log4j.Logger LOGGER =
        org.apache.log4j.Logger.getLogger(ExportPlugin.class);
    /**
     * Font property
     */
    public static final String NO_EMEDED_FONTS_PROPERTY =
        "de.renew.io.exportFormats.svg-no-embeded-fonts";
    private EPSExportFormat _epsExport;
    private PDFExportFormat _pdfExport;
    private SVGExportFormat _svgExport;
    private PNGExportFormat _pngExport;
    /**
     * pageSize property
     */
    public static final String PAGE_SIZE_PROP_NAME = "de.renew.io.export.pageSize";
    /**
     * pageOrientation property
     */
    public static final String PAGE_ORIENTATION_PROP_NAME = "de.renew.io.export.pageOrientation";
    /**
     * pageMargins property
     */
    public static final String PAGE_MARGINS_PROP_NAME = "de.renew.io.export.pageMargins";
    /**
     * epsFontHandling property
     */
    public static final String EPS_FONT_HANDLING_PROP_NAME = "de.renew.io.export.epsFontHandling";
    /**
     * Font handling "embed" value
     */
    public static final String EPS_FONT_HANDLING_EMBED = "embed";
    /**
     * Font handling "none" value
     */
    public static final String EPS_FONT_HANDLING_NONE = "none";
    /**
     * Font handling "shapes" value
     */
    public static final String EPS_FONT_HANDLING_SHAPES = "shapes";
    /**
     * Font handling value Array of properties
     */
    public static final String[] EPS_FONT_HANDLING_PROP_VALUES =
        new String[] { EPS_FONT_HANDLING_EMBED, EPS_FONT_HANDLING_NONE, EPS_FONT_HANDLING_SHAPES };
    /**
     * eps-transparency property
     */
    private static final String EPS_TRANSPARENCY_PROP_NAME = "de.renew.io.export.eps-transparency";
    /**
     * Bounding Box "BoundingBox" value
     */
    public static final String BOUNDING_BOX_PAGE_SIZE = "BoundingBox";

    /**
     * Creates an ExportPlugin with a parsed URL.
     *
     * @param location path to plugin.
     * @throws PluginException if URL is malformed or the configuration could not be loaded.
     */
    public ExportPlugin(URL location) throws PluginException {
        super(location);
    }

    /**
     * Creates an ExportPlugin with given PluginProperties.
     *
     * @param props the plugin configuration.
     */
    public ExportPlugin(PluginProperties props) {
        super(props);
    }

    /* (non-Javadoc)
     * @see de.renew.plugin.PluginAdapter#cleanup()
     */
    @Override
    public boolean cleanup() {
        StorableApi.removeExportFormat(_epsExport);
        StorableApi.removeExportFormat(_pdfExport);
        StorableApi.removeExportFormat(_svgExport);
        StorableApi.removeExportFormat(_pngExport);
        PluginManager.getInstance().removeCLCommand("ex");
        return true;
    }

    /*
     * (non-Javadoc)
     *
     * @see de.renew.plugin.PluginAdapter#getAlias()
     */
    @Override
    public String getAlias() {
        return "export";
    }

    /* (non-Javadoc)
     * @see de.renew.plugin.PluginAdapter#init()
     */
    @Override
    public void init() {
        _epsExport = new EPSExportFormat(getProperties());
        _pdfExport = new PDFExportFormat(getProperties());
        _svgExport = new SVGExportFormat();
        _pngExport = new PNGExportFormat(getProperties());
        StorableApi.addExportFormat(_epsExport);
        StorableApi.addExportFormat(_pdfExport);
        StorableApi.addExportFormat(_svgExport);
        StorableApi.addExportFormat(_pngExport);
        PluginManager.getInstance().addCLCommand("ex", new ExportClCommand());
    }

    /**
     * Returns the ExportPlugin if loaded in PluginManager.
     *
     * @return ExportPlugin if existing, otherwise returns null.
     */
    public static ExportPlugin getCurrent() {
        // Iterator it = PluginManager.getInstance().getPlugins().iterator();
        Iterator<IPlugin> it =
            PluginManager.getInstance().getPluginsProviding("de.renew.io.export").iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if (o instanceof ExportPlugin) {
                return (ExportPlugin) o;
            }
        }
        return null;
    }

    /**
     * Returns the Page size of the current bounding box.
     *
     * @return String containing the page size
     */
    public String getPageSize() {
        String result = ExportPlugin.BOUNDING_BOX_PAGE_SIZE;
        String userValue =
            getProperties().getProperty(ExportPlugin.PAGE_SIZE_PROP_NAME, PageConstants.A4);
        LOGGER.debug("Page Size user = " + userValue);
        for (String s : PageConstants.getSizeList()) {
            if (s.equalsIgnoreCase(userValue)) {
                result = s;
                getProperties().setProperty(ExportPlugin.PAGE_SIZE_PROP_NAME, s);
                break;
            }
        }
        if (userValue.equalsIgnoreCase(ExportPlugin.BOUNDING_BOX_PAGE_SIZE)) {
            result = ExportPlugin.BOUNDING_BOX_PAGE_SIZE;
            getProperties()
                .setProperty(ExportPlugin.PAGE_SIZE_PROP_NAME, ExportPlugin.BOUNDING_BOX_PAGE_SIZE);
        }
        LOGGER.debug("Page Size = " + result);
        return result;
    }

    /**
     * Returns the orientation of the current page.
     *
     * @return String containing the page orientation
     */
    public String getPageOrientation() {
        String result = PageConstants.PORTRAIT;
        String userValue = getProperties()
            .getProperty(ExportPlugin.PAGE_ORIENTATION_PROP_NAME, PageConstants.PORTRAIT);
        LOGGER.debug("Page orientation user = " + userValue);
        if (userValue.equalsIgnoreCase(PageConstants.PORTRAIT)) {
            result = PageConstants.PORTRAIT;
            getProperties().setProperty(ExportPlugin.PAGE_ORIENTATION_PROP_NAME, result);
        } else if (userValue.equalsIgnoreCase(PageConstants.LANDSCAPE)) {
            result = PageConstants.LANDSCAPE;
            getProperties().setProperty(ExportPlugin.PAGE_ORIENTATION_PROP_NAME, result);
        }
        LOGGER.debug("Page orientation = " + result);
        return result;
    }

    /**
     * Returns the current EPS font shapes.
     *
     * @return String containing the current font shapes
     */
    public String getEpsFontHandling() {
        String result = EPS_FONT_HANDLING_SHAPES;
        String userValue =
            getProperties().getProperty(EPS_FONT_HANDLING_PROP_NAME, EPS_FONT_HANDLING_SHAPES);
        LOGGER.debug("EPS font handling user = " + userValue);
        for (String s : EPS_FONT_HANDLING_PROP_VALUES) {
            if (s.equalsIgnoreCase(userValue)) {
                result = s;
                getProperties().setProperty(EPS_FONT_HANDLING_PROP_NAME, result);
                break;
            }
        }
        LOGGER.debug("EPS font handling = " + result);
        return result;
    }

    /**
     * Checks if Eps transparent.
     *
     * @return boolean true if Eps is transparent
     */
    public boolean getEpsTransparency() {
        boolean result = getProperties().getBoolProperty(EPS_TRANSPARENCY_PROP_NAME);
        LOGGER.debug("EPS tansparency user = " + result);
        return result;
    }
}
