package de.renew.console.completer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import jline.console.completer.Completer;

import static jline.internal.Preconditions.checkNotNull;


/**
 * Completer which contains multiple completers and aggregates them together.
 *
 * @author <a href="mailto:jason@planet57.com">Jason Dillon</a>
 * @since 2.3
 * @author cabac
 *
 */
public class CompleterComposition implements Completer {
    private final List<Completer> _completers = new ArrayList<Completer>();

    /**
     * Constructs an empty {@code CompleterComposition}.
     */
    public CompleterComposition() {
        // empty
    }

    /**
     * Construct an AggregateCompleter with the given collection of completers.
     * The completers will be used in the iteration order of the collection.
     *
     * @param completers the collection of completers
     */
    public CompleterComposition(final Collection<Completer> completers) {
        checkNotNull(completers);
        this._completers.addAll(completers);
    }

    /**
     * Construct an AggregateCompleter with the given completers.
     * The completers will be used in the order given.
     *
     * @param completers the completers
     */
    public CompleterComposition(final Completer... completers) {
        this(Arrays.asList(completers));
    }

    /**
     * Retrieve the collection of completers currently being aggregated.
     *
     * @return the aggregated completers
     */
    public Collection<Completer> getCompleters() {
        return _completers;
    }

    /**
     * Perform a completion operation across all aggregated completers.
     *
     * @see Completer#complete(String, int, java.util.List)
     * @return the highest completion return value from all completers
     */
    @Override
    public int complete(
        final String buffer, final int cursor, final List<CharSequence> candidates)
    {
        // buffer could be null
        checkNotNull(candidates);

        List<Completion> completions = new ArrayList<Completion>(_completers.size());

        // Run each completer, saving its completion results
        int max = -1;
        for (Completer completer : _completers) {
            Completion completion = new Completion(candidates);
            completion.complete(completer, buffer, cursor);

            // Compute the max cursor position
            max = Math.max(max, completion.getCursor());

            completions.add(completion);
        }

        // Append candidates from completions which have the same cursor position as max
        for (Completion completion : completions) {
            if (completion.getCursor() == max) {
                candidates.addAll(completion.getCandidates());
            }
        }

        return max;
    }

    /**
     * @return a string representing the aggregated completers
     */
    @Override
    public String toString() {
        return getClass().getSimpleName() + "{" + "completers=" + _completers + '}';
    }

    /**
     * A helper class to hold the result of a completion operation performed by a completer.
     */
    private class Completion {
        private final List<CharSequence> _candidates;
        private int _cursor;

        /**
         * Constructs a new {@code Completion} object with the provided candidates list.
         *
         * @param candidates the candidates to complete
         * @throws NullPointerException if {@code candidates} is {@code null}
         */
        Completion(final List<CharSequence> candidates) {
            checkNotNull(candidates);
            _candidates = new LinkedList<CharSequence>(candidates);
        }

        /**
         * Returns the list of candidates completed by the completer.
         *
         * @return the list of completion candidates
         */
        public List<CharSequence> getCandidates() {
            return _candidates;
        }

        /**
         * Returns the cursor position after the completion.
         *
         * @return the cursor position after completion
         */
        public int getCursor() {
            return _cursor;
        }

        /**
         * Performs the completion for a specific completer, updating the list of candidates
         * and the cursor position.
         *
         * @param completer the completer to use for the operation
         * @param buffer the input buffer for completion
         * @param cursor the current cursor position
         */
        public void complete(final Completer completer, final String buffer, final int cursor) {
            checkNotNull(completer);
            this._cursor = completer.complete(buffer, cursor, _candidates);
        }
    }

    /**
     * Adds a new completer to this composition.
     *
     * @param completer the completer to add
     */
    public void addCompleter(Completer completer) {
        _completers.add(completer);
    }

    /**
     * Removes a completer from this composition.
     *
     * @param completer the completer to remove
     */
    public void removeCompleter(Completer completer) {
        _completers.remove(completer);
    }
}