/*
 * @(#)Handle.java 5.1
 *
 */

package de.renew.draw.ui.ontology;

import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import de.renew.draw.storables.ontology.Figure;

/**
 * Handles are used to change a figure by direct manipulation.
 * Handles know their owning figure, and they provide methods to
 * locate the handle on the figure and to track changes.
 * <hr>
 * <b>Design Patterns</b><P>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld004.htm>Adapter</a></b><br>
 * Handles adapt the operations to manipulate a figure to a common interface.
 *
 * @see Figure
 */
public interface FigureHandle {
    /**
     * Default size (in pixels) of a UI handle used for resizing or interaction.
     */
    int HANDLESIZE = 8;

    /**
     * Locates the handle on the figure. The handle is drawn
     * centered around the returned point.
     * @return point of handle of figure
     */
    Point locate();

    /**
     * Tracks the start of the interaction. The default implementation
     * does nothing.
     * @param e the mouse event which started the interaction
     * @param x the x position where the interaction started
     * @param y the y position where the interaction started
     * @param view the handles container
     */
    void invokeStart(MouseEvent e, int x, int y, DrawingView view);

    /**
     * Tracks a step of the interaction.
     * @param e the mouse event which started the interaction
     * @param x the current x position
     * @param y the current y position
     * @param anchorX the x position where the interaction started
     * @param anchorY the y position where the interaction started
     * @param view an instance of DrawingView
     */
    void invokeStep(MouseEvent e, int x, int y, int anchorX, int anchorY, DrawingView view);

    /**
     * Tracks the end of the interaction.
     * @param e the mouse event which started the interaction
     * @param x the current x position
     * @param y the current y position
     * @param anchorX the x position where the interaction started
     * @param anchorY the y position where the interaction started
     * @param view an instance of DrawingView
     */
    void invokeEnd(MouseEvent e, int x, int y, int anchorX, int anchorY, DrawingView view);

    /**
     * Handles clean-up whenever the user
     * aborts a handle (before completion).
     * @param view the drawing view from which to clean-up
     */
    void abort(DrawingView view);

    /**
     * Gets the handle's owner.
     * @return handle's owner
     */
    Figure owner();

    /**
     * Gets the display box of the handle.
     * @return display box of handle
     */
    Rectangle displayBox();

    /**
     * Tests if a point is contained in the handle.
     * @param x point on x-axis
     * @param y point on y-axis
     * @return boolean, {@code true} if given point is contained in handle
     */
    boolean containsPoint(int x, int y);

    /**
     * Draws this handle.
     * @param g graphic to be drawn in
     */
    void draw(Graphics g);
}