/*
 * @(#)DrawingView.java 5.1
 *
 */

package de.renew.draw.ui.ontology;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.geom.AffineTransform;
import java.awt.image.ImageObserver;
import java.awt.print.Printable;
import java.util.Collection;
import java.util.Vector;

import de.renew.draw.storables.ontology.Drawing;
import de.renew.draw.storables.ontology.DrawingChangeListener;
import de.renew.draw.storables.ontology.Figure;
import de.renew.draw.storables.ontology.FigureEnumeration;
import de.renew.draw.ui.api.EditorApi;
import de.renew.draw.ui.api.SelectionApi;

/**
 * DrawingView renders a Drawing and listens to its changes.
 * It receives user input and delegates it to the current tool.
 * <hr>
 * <b>Design Patterns</b><P>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld026.htm>Observer</a></b><br>
 * DrawingView observes drawing for changes via the DrawingListener interface.<br>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld032.htm>State</a></b><br>
 * DrawingView plays the role of the StateContext in
 * the State pattern. Tool is the State.<br>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld034.htm>Strategy</a></b><br>
 * DrawingView is the StrategyContext in the Strategy pattern
 * with regard to the UpdateStrategy. <br>
 * DrawingView is the StrategyContext for the PointConstrainer.
 * <hr>
 *
 *  See {@link Drawing}, {@link Painter}, {@link Tool}.
 */
public interface DrawingView extends ImageObserver, DrawingChangeListener, Printable {

    /**
     * Gets the drawing.
     * @return current drawing
     */
    Drawing drawing();

    /**
     * Sets and installs another drawing in the view.
     * @param d drawing to be set/installed
     */
    void setDrawing(Drawing d);

    /**
     * Gets the editor.
     * @return editor
     */
    DrawingEditor editor();

    /**
     * Adds a figure to the drawing.
     * @param figure figure to added
     * @return the added figure
     */
    Figure add(Figure figure);

    /**
     * Removes a figure from the drawing.
     * @param figure vector of figures to be removed
     * @return the removed figure
     */
    Figure remove(Figure figure);

    /**
     * Adds a vector of figures to the drawing.
     * @param figures vector of figures to be removed
     */
    void addAll(Vector<Figure> figures);

    /**
     * Gets the size of the drawing.
     * @return size of this drawing
     */
    Dimension getSize();

    /**
     * Gets the currently selected figures.
     * @return a vector with the selected figures. The vector
     * is a copy of the current selection.
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#getSelectedFigures()} instead.
     */
    @Deprecated
    Vector<Figure> selection();

    /**
     * Gets an enumeration over the currently selected figures.
     * @return enumeration of selected figures
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#getSelectedFigures()} instead.
     */
    @Deprecated
    FigureEnumeration selectionElements();

    /**
     * Gets the currently selected figures in Z order.
     * @return a vector with the selected figures. The vector
     * is a copy of the current selection.
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#getSelectedFiguresZOrdered()} instead.
     */
    @Deprecated
    Vector<Figure> selectionZOrdered();

    /**
     * Gets the number of selected figures.
     * @return number of selected figures
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#getSelectionCount()} instead.
     */
    @Deprecated
    int selectionCount();

    /**
     * Adds a figure to the current selection.
     * @param figure figure to be added into selection
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#addToSelection(Figure)} instead.
     */
    @Deprecated
    void addToSelection(Figure figure);

    /**
     * Adds an enumeration of figures to the current selection.
     * @param figures figures to be added
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#addToSelectionAll(FigureEnumeration)} instead.
     */
    @Deprecated
    void addToSelectionAll(FigureEnumeration figures);

    /**
     * Adds a vector of figures to the current selection.
     * @param figures figures included in vector to be added
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#addToSelectionAll(Collection)} instead.
     */
    @Deprecated
    void addToSelectionAll(Vector<Figure> figures);

    /**
     * Removes a vector of figures from the current selection.
     * @param figures figures included in vector to be removed
     */
    void removeFromSelectionAll(Vector<Figure> figures);

    /**
     * If a figure isn't selected it is added to the selection.
     * Otherwise, it is removed from the selection.
     * @param figure figure added if selected, otherwise removed
     */
    void toggleSelection(Figure figure);

    /**
     * Clears the current selection.
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#clearSelection()} instead.
     */
    @Deprecated
    void clearSelection();

    /**
     * Toggles a vector of figures.
     * If a figure isn't selected it is added to the selection.
     * Otherwise, it is removed from the selection.
     * @param figures figures vector added if selected, otherwise removed
     */
    void toggleSelectionAll(Vector<Figure> figures);

    /**
     * Invalidates the handles of the current selection.
     * This means that the cached set of handles will be
     * re-calculated next time the selection's handles are
     * queried.
     * @deprecated This method is not to be used externally. Please use the method {@link SelectionApi#selectionInvalidateHandles()} instead.
     */
    @Deprecated
    void selectionInvalidateHandles();

    /**
     * Gets the current selection as a FigureSelection. A FigureSelection
     * can be cut, copied, pasted.
     * @return current selection
     */
    FigureSelection getFigureSelection();

    /**
     * Finds a handle at the given coordinates.
     * @param x point on x-axis
     * @param y point on y-axis
     * @return the hit handle, <code>null</code> if no handle is found
     */
    FigureHandle findHandle(int x, int y);

    /**
     * Gets the position of the last click inside the view.
     * @return position of last click inside view
     */
    Point lastClick();

    /**
     * Gets the position of the last mouse event inside the view.
     * @return current position of mouse inside view
     */
    Point getCurrentMousePosition();

    /**
     * Sets the current point constrainer.
     * @param p point to be set
     */
    void setConstrainer(PointConstrainer p);

    /**
     * Gets the current constrainer.
     * @return the currently active PointConstrainer, <code>null</code> if none is set
     */
    PointConstrainer getConstrainer();

    /**
     * Checks whether the drawing has some accumulated damage
     * and informs all views about the required update,
     * if neccessary.
     * @deprecated This method is not to be used externally. Please use the method {@link EditorApi#checkDamage()} instead.
     */
    @Deprecated
    void checkDamage();

    /**
     * Repair the damaged area of this view.
     * @deprecated This method is not to be used externally. Please use the method {@link EditorApi#repairDamage()} instead.
     */
    @Deprecated
    void repairDamage();

    /**
     * Creates an image with the given dimensions.
     * @param width width of image
     * @param height height of image
     * @return created image with given parameters
     */
    Image createImage(int width, int height);

    /**
     * Gets a graphic to draw into.
     * @return graphic
     */
    Graphics getGraphics();

    /**
     * Gets the background color of the DrawingView.
     * @return background color
     */
    Color getBackground();

    /**
     * Sets the background color of the DrawingView.
     * @param c new background color
     */
    void setBackground(Color c);

    /**
     * Draws the contents of the drawing view.
     * The view has three layers: background, drawing, handles.
     * The layers are drawn in back to front order.
     * @param g graphic to be drawn into
     */
    void drawAll(Graphics g);

    /**
     * Sets the cursor of the DrawingView.
     * @param c cursor to be set
     */
    void setCursor(Cursor c);

    /**
     * Freezes the view by acquiring the drawing lock.
     * See {@link Drawing#lock()}.
     */
    void freezeView();

    /**
     * Unfreezes the view by releasing the drawing lock.
     * See {@link Drawing#unlock()}.
     */
    void unfreezeView();

    /**
     * Moves Selection by given parameters.
     * @param dx x amount on x-axis
     * @param dy y amount on y-axis
     */
    void moveSelection(int dx, int dy);

    /**
     * Moves the view of a ScrollPane, so the given figure
     * will be visible.
     * @param fig figure to be made visible
     */
    void showElement(Figure fig);

    /**
     * Get the transformation of this view's graphics.
     *
     * @return the {@link AffineTransform} object affecting this views graphics
     */
    AffineTransform getAffineTransform();
}