package de.renew.draw.ui.api.services;


import java.io.File;

import de.renew.draw.ui.ontology.dialogs.InputDialogBuilder;
import de.renew.draw.ui.ontology.dialogs.MessageDialogBuilder;
import de.renew.draw.ui.ontology.dialogs.OptionDialogBuilder;
import de.renew.ioontology.FileFilter;

/**
 * Service class that provides methods for the {@link de.renew.draw.ui.api.DialogApi} to call.
 */
public interface DialogService {
    /**
     * Opens a file chooser dialog to let the user choose the path where they want to save the file
     *
     * @param file the initial file name/path where to start the dialog, if it is null first tries to get
     *             the file name of the current drawing. If that does not work the lastPath is used.
     * @param filter the file filter for the JFileChooser
     * @return the chosen file or null if the filter was null
     */
    File showSaveFileDialog(File file, FileFilter filter);

    /**
     * Opens a file chooser dialog to select the file, which will be imported/opened.
     *
     * @param filter the FileFilter, which is used
     * @return File[] an array of files to be imported
     */
    File[] showSelectFilesDialog(FileFilter filter);

    /**
     * Opens a file dialog and returns the selected/entered file.
     * The user will not be able to select more than one file.
     * The file name is checked and completed with one of the default Extensions.
     * If the file name is prefixed with a directory, the dialog initially switches
     * its actual directory to this prefix. In the other case, the actual directory is
     * set to the last directory used for files with the file's extension.
     *
     * @param file the file name that will be proposed to the user.
     * @param filter the FileFilter that determines the selectable file types in the FileChooser
     * @return the chosen File to which the default extension is appended, returns null, if the dialog was cancelled.
     **/
    File showSelectFilesDialog(File file, FileFilter filter);

    /**
     * Opens a file dialog and returns the selected/entered file.
     * The user will be able to select more than one file if multiSelection is true.
     * The file name is checked and completed with one of the default Extensions.
     * If the file name is prefixed with a directory, the dialog initially switches
     * its actual directory to this prefix. In the other case, the actual directory is
     * set to the last directory used for files with the file's extension.
     *
     * @param file the file name that will be proposed to the user.
     * @param filter the FileFilter that determines the selectable file types in the FileChooser
     * @param multiSelection boolean indicating whether it is allowed to choose multiple files
     * @return the chosen File to which the default extension is appended, returns null, if the dialog was cancelled.
     **/
    File[] showSelectFilesDialog(File file, FileFilter filter, boolean multiSelection);

    /**
     * Creates a new builder for the message dialog.
     * @return a new {@link MessageDialogBuilder}
     */
    MessageDialogBuilder createMessageDialogBuilder();

    /**
     * Creates a new builder for the input dialog.
     * @return a new {@link InputDialogBuilder}
     */
    InputDialogBuilder createInputDialogBuilder();

    /**
     * Creates a new builder for the option dialog.
     * @return a new {@link OptionDialogBuilder}
     */
    OptionDialogBuilder createOptionDialogBuilder();
}
