package de.renew.draw.ui.api;

import java.util.Collection;
import java.util.Vector;

import de.renew.draw.storables.ontology.Figure;
import de.renew.draw.storables.ontology.FigureEnumeration;
import de.renew.draw.ui.api.services.SelectionService;
import de.renew.draw.ui.impl.services.SelectionServiceImpl;
import de.renew.draw.ui.ontology.ErrorState;

/**
 * API class for getting and adding figures that are currently selected as well as the
 * number of all currently selected figures.
 */
public final class SelectionApi {

    private static SelectionService _service;

    // The static initialization that creates the service object
    static {
        _service = new SelectionServiceImpl();
    }

    /**
     * Gets the currently selected figures.
     * @return a vector with the selected figures, the vector is a copy of the current selection
     */
    public static Vector<Figure> getSelectedFigures() {
        return _service.getSelectedFigures();
    };

    /**
     * Gets the currently selected figures in Z order, which means the figures are given bottom to top of the
     * current view. The Z stands for the z-axis of the current view.
     * @return a vector with the selected figures
     */
    public static Vector<Figure> getSelectedFiguresZOrdered() {
        return _service.getSelectedFiguresZOrdered();
    };

    /**
     * Adds a figure to the current selection.
     * @param figure figure to be added into selection
     */
    public static void addToSelection(Figure figure) {
        _service.addToSelection(figure);
    };

    /**
     * Adds an enumeration of figures to the current selection.
     * @param figureEnumeration figures to be added
     */
    public static void addToSelectionAll(FigureEnumeration figureEnumeration) {
        _service.addToSelectionAll(figureEnumeration);
    }

    /**
     * Adds a vector of figures to the current selection.
     * @param figures figures included in vector to be added
     */
    public static void addToSelectionAll(Collection<Figure> figures) {
        _service.addToSelectionAll(figures);
    }

    /**
     * Invalidates the handles of the current selection.
     * This means that the cached set of handles will be
     * re-calculated next time the selection's handles are
     * queried.
     */
    public static void selectionInvalidateHandles() {
        _service.selectionInvalidateHandles();
    }

    /**
     * Selects the corresponding elements for the given FigureException.
     *
     * @param errorState the FigureException whose offending elements are searched
     * @return the offending elements for the exception
     */
    public static boolean selectOffendingElements(ErrorState errorState) {
        return _service.selectOffendingElements(errorState);
    }

    /**
     * Gets the number of selected figures.
     * @return number of selected figures
     */
    public static int getSelectionCount() {
        return _service.getSelectionCount();
    }

    /**
     * Clears the current selection.
     */
    public static void clearSelection() {
        _service.clearSelection();
    }

}
