package de.renew.draw.ui.api;

import java.io.File;

import de.renew.draw.ui.impl.services.DialogServiceImpl;
import de.renew.draw.ui.ontology.dialogs.InputDialogBuilder;
import de.renew.draw.ui.ontology.dialogs.MessageDialogBuilder;
import de.renew.draw.ui.ontology.dialogs.OptionDialogBuilder;
import de.renew.ioontology.FileFilter;

/**
 * Service class to open file chooser dialogs to input the path to save/load a file.
 */
public final class DialogApi {
    private static DialogServiceImpl _service;

    static {
        _service = new DialogServiceImpl();
    }

    private DialogApi() {}

    /**
     * Opens a file chooser dialog to let the user choose the path where they want to save the file
     *
     * @param file the initial file name/path where to start the dialog, if it is null first tries to get
     *             the filename of the current drawing. If that does not work the lastPath is used.
     * @param filter the file filter for the JFileChooser
     * @return the chosen file or null if the filter was null
     */
    public static File showSaveFileDialog(File file, FileFilter filter) {
        return _service.showSaveFileDialog(file, filter);
    }

    /**
     * Opens a file chooser dialog to select the file, which will be imported/opened.
     *
     * @param filter the FileFilter, which is used
     * @return File[] an array of files to be imported
     */
    public static File[] showSelectFilesDialog(FileFilter filter) {
        return _service.showSelectFilesDialog(filter);
    }

    /**
     * Opens a file dialog and returns the selected/entered file.
     * The user will not be able to select more than one file.
     * The file name is checked and completed with one of the default Extensions.
     * If the file name is prefixed with a directory, the dialog initially switches
     * its actual directory to this prefix. In the other case, the actual directory is
     * set to the last directory used for files with the file's extension.
     *
     * @param file the file name that will be proposed to the user.
     * @param filter the FileFilter that determines the selectable file types in the FileChooser
     * @return the chosen File to which the default extension is appended, returns null, if the dialog was cancelled.
     **/
    public static File showSelectFilesDialog(File file, FileFilter filter) {
        return _service.showSelectFilesDialog(file, filter);

    }

    /**
     * Opens a file dialog and returns the selected/entered file.
     * The user will be able to select more than one file if multiSelection is true.
     * The file name is checked and completed with one of the default Extensions.
     * If the file name is prefixed with a directory, the dialog initially switches
     * its actual directory to this prefix. In the other case, the actual directory is
     * set to the last directory used for files with the file's extension.
     *
     * @param file the file name that will be proposed to the user.
     * @param filter the FileFilter that determines the selectable file types in the FileChooser
     * @param multiSelection boolean indicating whether it is allowed to choose multiple files
     * @return the chosen File to which the default extension is appended, returns null, if the dialog was cancelled.
     **/
    public static File[] showSelectFilesDialog(
        File file, FileFilter filter, boolean multiSelection)
    {
        return _service.showSelectFilesDialog(file, filter, multiSelection);

    }

    /**
     * Creates a new builder for the message dialog.
     * @return a new {@link MessageDialogBuilder}
     */
    public static MessageDialogBuilder createMessageDialogBuilder() {
        return _service.createMessageDialogBuilder();
    }

    /**
     * Creates a new builder for the input dialog.
     * @return a new {@link InputDialogBuilder}
     */
    public static InputDialogBuilder createInputDialogBuilder() {
        return _service.createInputDialogBuilder();
    }

    /**
     * Creates a new builder for the option dialog.
     * @return a new {@link OptionDialogBuilder}
     */
    public static OptionDialogBuilder createOptionDialogBuilder() {
        return _service.createOptionDialogBuilder();
    }

}
