package de.renew.draw.ui.api;

import java.io.File;
import java.util.Properties;

import de.renew.draw.ui.impl.services.ApplicationServiceImpl;

/**
 * Service class that provides application-wide services and behaviors for the Renew drawing environment, such as applying
 * drawing properties, managing save paths, showing status messages, and registering drawing view factories.
 * @author Percy Hohberg
 */
public final class ApplicationApi {

    private static final ApplicationServiceImpl APPLICATION_SERVICE;

    static {
        APPLICATION_SERVICE = new ApplicationServiceImpl();
    }

    /**
     * Sets behavior of the drawing via their properties.
     * @param properties the properties to set the gui to
     */
    public static void setDrawProperties(Properties properties) {
        APPLICATION_SERVICE.setDrawProperties(properties);
    }

    /**
     * Finds out if a frame can be reshaped.
     * @return <code>true</code> if frames may be positioned and resized by user
     **/
    public static boolean isAvoidFrameReshape() {
        return APPLICATION_SERVICE.isAvoidFrameReshape();
    }

    /**
     * Finds out if graphics are used.
     * @return <code>true</code> if a drawing will be loaded without graphics
     **/
    public static boolean isNoGraphics() {
        return APPLICATION_SERVICE.isNoGraphics();
    }

    /**
     * Getter for last save path.
     * @return last save path
     */
    public static File getLastPath() {
        return APPLICATION_SERVICE.getLastPath();
    }

    /**
     * Verifies that closing the editor is possible.
     * @return <code>true</code> if it is possible to close the editor
     */
    public static boolean canClose() {
        return APPLICATION_SERVICE.canClose();
    }

    /**
     * Requests closing of the editor.
     */
    public static void requestClose() {
        APPLICATION_SERVICE.requestClose();
    }

    /**
     * Displays a status message in the gui status line.
     * @param message status that is displayed
     */
    public static void showStatus(String message) {
        APPLICATION_SERVICE.showStatus(message);
    }


    /**
     * Register a factory ID. When a new drawing is created, the factory with the given ID is used.
     * @param key Use the name of the drawing as a key
     * @param id The id of the factory registered at the Workbench
     */
    public static void registerDrawingViewFactory(Class<?> key, String id) {
        APPLICATION_SERVICE.registerDrawingViewFactory(key, id);
    }
}
