package de.renew.draw.storables.ontology;

import java.io.IOException;

/**
 * An input stream that can be used to resurrect {@link de.renew.draw.storables.ontology.Storable} objects.
 * StorableInput preserves the object identity of the stored objects.
 */
public interface StorableInput extends StorableInOut {
    /**
     * Gets the version of the StorableInput.
     *
     * @return an {@code int}, which describes the used version
     */
    int getVersion();

    /**
     * Sets the version of the StorableInput.
     *
     * @param version the new version of the StorableInput
     */
    void setVersion(int version);

    /**
     * Reads and resurrects a Storable object from the input stream.
     *
     * @return the {@code Storable} that has been read from the input stream or<br>
     *         {@code null} if it wasn't successfully read from the StreamTokenizer
     * @throws IOException throws IOException if an I/O error occurred while reading from the input
     */
    Storable readStorable() throws IOException;

    /**
     * Reads a string from the input stream.
     *
     * @return the next token from the StreamTokenizer as String
     * @throws IOException throws IOException if the next token from the StreamTokenizer isn't a String
     */
    String readString() throws IOException;

    /**
     * Determines whether an int could be read from the input stream.
     *
     * @return {@code true}, if int can be read successfully
     * @throws IOException throws IOException if the next token from the StreamTokenizer isn't an int
     */
    boolean canReadInt() throws IOException;

    /**
     * Reads an int from the input stream.
     *
     * @return the int that has been read from the input stream
     * @throws IOException throws IOException if the next token from the StreamTokenizer isn't an int
     */
    int readInt() throws IOException;

    /**
     * Reads a double from the input stream.
     *
     * @return the double read from the input stream
     * @throws IOException throws IOException if the next token from the StreamTokenizer isn't a double
     */
    double readDouble() throws IOException;

    /**
     * Reads a boolean from the input stream.
     *
     * @return the boolean from the input stream
     * @throws IOException throws IOException if an I/O error occurs while reading from the input
     */
    boolean readBoolean() throws IOException;

    /**
     * Closes the storable input stream.
     */
    void close();


    /**
     * Puts the token back.
     */
    void putBack();
}
