package CH.ifa.draw.standard;

import java.util.ArrayList;
import java.util.List;
import javax.swing.JMenuItem;

import CH.ifa.draw.application.DrawApplication;
import de.renew.draw.storables.ontology.Drawing;
import de.renew.draw.storables.ontology.DrawingListener;
import de.renew.draw.ui.api.services.MenuService;
import de.renew.draw.ui.impl.services.MenuServiceImpl;

/**
 * WindowMenuManager is like a "hook" for the WindowsMenuCreator,
 * as the application does not exist yet when the DrawingListeners are registered.
 */
public class WindowsMenuManager {
    /**
     * Current drawApplication, provides access to components of the Renew Application.
     */
    static private DrawApplication _drawApplication;
    /**
     * If the drawApplication does not exist yet, the pending DrawingListeners will be added
     * as soon as the drawingApplication exists by adding all entries from this list.
     */
    private static final List<DrawingListener> PENDING_LISTENERS = new ArrayList<>();

    /**
     * Registers the drawApplication to a field variable, so the WindowsMenuManager can access its components.
     * @param drawApplication the drawApplication that is written on to the field variable
     */
    public static void setGui(DrawApplication drawApplication) {
        if (drawApplication != null) {
            _drawApplication = drawApplication;
            for (DrawingListener l : PENDING_LISTENERS) {
                _drawApplication.addDrawingListener(l);
            }
            PENDING_LISTENERS.clear();
        }
    }

    /**
     * Registers a drawingListener to the drawApplication.
     * @param drawingListener that is registered.
     */
    public static void addDrawingListener(DrawingListener drawingListener) {
        if (_drawApplication != null) {
            _drawApplication.addDrawingListener(drawingListener);
        } else {
            PENDING_LISTENERS.add(drawingListener);
        }
    }

    /**
     * Registers a new MenuItem to the WindowsMenu.
     * @param toAdd item menu that is added
     */
    public static void addMenuItem(JMenuItem toAdd) {
        MenuService menuService = new MenuServiceImpl();
        menuService.registerMenu("Windows ", toAdd);

    }

    /**
     * Removes a MenuItem from the WindowsMenu.
     * @param toRemove item menu to remove
     */
    public static void removeMenuItem(JMenuItem toRemove) {
        MenuService menuService = new MenuServiceImpl();
        menuService.unregisterMenu(toRemove);
    }

    /**
     * Brings the DrawingViewContainer of a drawing to the foreground.
     * @param drawing drawing that is supposed to be focused.
     */
    public static void requestFocusForDrawing(de.renew.draw.storables.ontology.Drawing drawing) {
        _drawApplication.showDrawingViewContainer(drawing);
    }

    /**
     * Adds a FocusListener to the drawApplication.
     * @param windowsMenuCreator that is registered to the drawApplication
     */
    public static void addFocusListenerToDrawing(WindowsMenuCreator windowsMenuCreator) {
        _drawApplication.registerFocusListener(windowsMenuCreator);
    }

    /**
     * Returns the drawing that is currently focused.
     * @return drawing that is currently focused
     */
    public static Drawing askForCurrentlyFocusedDrawing() {
        return _drawApplication.drawing();
    }

    /**
     * Adds a SavedListener to the drawApplication
     * @param windowsMenuCreator that is registered to the drawApplication
     */
    public static void addSavedListener(WindowsMenuCreator windowsMenuCreator) {
        _drawApplication.addSavedListener(windowsMenuCreator);
    }
}
