/*
 * @(#)AbstractTool.java 5.1
 *
 */

package CH.ifa.draw.standard;

import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.net.URI;

import de.renew.draw.storables.ontology.Drawing;
import de.renew.draw.ui.ontology.DrawingEditor;
import de.renew.draw.ui.ontology.DrawingView;
import de.renew.draw.ui.ontology.Tool;

/**
 * Default implementation support for Tools.
 *
 * @see DrawingView
 * @see Tool
 */
public class AbstractTool implements Tool {
    /**
     * The mediator for the participants of
     * the drawing editor.
     */
    protected final DrawingEditor fEditor;

    /**
     * The position of the initial mouse down.
     */
    protected int _anchorX;

    /**
     * The position of the initial mouse down.
     */
    protected int _anchorY;

    /**
     * Constructs a tool for the given view.
     * @param editor the given view.
     */
    public AbstractTool(DrawingEditor editor) {
        fEditor = editor;
    }

    /**
     * Activates the tool for the given view. This method is called
     * whenever the user switches to this tool. Use this method to
     * reinitialize a tool.
     */
    @Override
    public void activate() {
        fEditor.view().clearSelection();
        fEditor.view().checkDamage();
    }

    /**
     * Deactivates the tool. This method is called whenever the user
     * switches to another tool. Use this method to do some clean-up
     * when the tool is switched. Subclassers should always call
     * super.deactivate.
     */
    @Override
    public void deactivate() {
        fEditor.view().setCursor(Cursor.getDefaultCursor());
    }

    /**
     * Handles mouse down events in the drawing view.
     */
    @Override
    public void mouseDown(MouseEvent e, int x, int y) {
        _anchorX = x;
        _anchorY = y;
    }

    /**
     * Handles mouse drag events in the drawing view.
     */
    @Override
    public void mouseDrag(MouseEvent e, int x, int y) {}

    /**
     * Handles mouse up in the drawing view.
     */
    @Override
    public void mouseUp(MouseEvent e, int x, int y) {}

    /**
     * Handles clean-up whenever the user
     * cancels a tool before completion.
     */
    @Override
    public void cancel() {
        editor().toolDone();
    }

    /**
     * Handles mouse moves (if the mouse button is up).
     */
    @Override
    public void mouseMove(MouseEvent evt, int x, int y) {}

    /**
     * Handles key down events in the drawing view.
     */
    @Override
    public void keyDown(KeyEvent evt, int key) {
        fEditor.dispatchEvent(evt);
    }

    /**
     * Gets the tool's drawing.
     * @return the tool's drawing.
     */
    public Drawing drawing() {
        return fEditor.drawing();
    }

    /**
     * Gets the tool's editor.
     * @return the tool's editor.
     */
    public DrawingEditor editor() {
        return fEditor;
    }

    /**
     * Gets the tool's view.
     * @return the tool's view.
     */
    public DrawingView view() {
        return fEditor.view();
    }

    /**
     * Provides a default implementation which draws nothing.
     * This should be appropriate for most tool implementations
     * since they use drawing figures to give visual feedback.
     **/
    @Override
    public void draw(Graphics g) {}

    /**
     * Provides the possibility to extend the tool's action.
     * (Activated by ctrl-click.)
     *
     * @param uri - argument of the action.
     */
    public void simAccess(URI uri) {}
}