package CH.ifa.draw.application;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;


/**
 * A dialog to choose the type of new drawing.
 * @author Michael Duvigneau
 * @since Renew 2.0
 **/
public class NewDrawingDialog extends JDialog {
    /**
     * buttonPanel is a JPanel for buttons
     */
    private JPanel buttonPanel;
    /**
     * cancelButton is a Button that closes the Dialog without creating a new drawing.
     */
    private JButton cancelButton;
    /**
     * okButton is a Button that creates a new drawing and then closes the dialog.
     */
    private JButton okButton;
    /**
     * Label for different drawing types.
     */
    private JLabel typeLabel;
    /**
     * List that contains all available drawing types.
     */
    private JList typeList;
    /**
     * Index of the selected item. It defaults to -1, if nothing is selected.
     */
    private int result = -1;

    /**
     * Creates a Drawing JDialog of the type that is chosen from an Array with a certain Index,
     * Shows a dialog window titled "Choose new drawing type", a list of options from the types array
     * and highlights the option at the selected Index. It also creates a "New" and "Cancel" Button to confirm/cancel.
     * @param parent parent Frame of this dialog
     * @param types Array that contains the types of dialogs
     * @param selectedIndex Index that tells us which of the types we choose out of the types[] array
     */

    public NewDrawingDialog(JFrame parent, String[] types, int selectedIndex) {
        super(parent, true);
        initComponents();
        typeList.setListData(types);
        if (selectedIndex < types.length && selectedIndex >= 0) {
            typeList.setSelectedIndex(selectedIndex);
        }
        pack();
    }

    private void initComponents() {
        buttonPanel = new JPanel();
        okButton = new JButton();
        cancelButton = new JButton();
        typeLabel = new JLabel();
        typeList = new JList();

        getContentPane().setLayout(new BorderLayout());

        setTitle("Choose new drawing type");
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent evt) {
                closeDialog();
            }
        });

        typeLabel.setText("Available drawing types:");
        getContentPane().add(typeLabel, BorderLayout.NORTH);

        typeList.setToolTipText("Choose the type of the drawing to create");
        getContentPane().add(typeList, BorderLayout.CENTER);
        typeList.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent e) {
                if (e.getClickCount() == 2) {
                    commitDialog();
                }
            }
        });

        okButton.setText("New");
        okButton.setToolTipText("Creates a new drawing of the chosen type.");
        okButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                commitDialog();
            }
        });
        buttonPanel.add(okButton);

        cancelButton.setToolTipText("Cancels the dialog without creating a drawing.");
        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                closeDialog();
            }
        });
        buttonPanel.add(cancelButton);

        getContentPane().add(buttonPanel, BorderLayout.SOUTH);
    }

    private void commitDialog() {
        result = typeList.getSelectedIndex();
        setVisible(false);
    }

    /**
     * Closes the dialog and unsets the index of the selected item in the list.
     */
    private void closeDialog() {
        result = -1;
        setVisible(false);
    }

    /**
     * Sets dialog to visible and returns index of the selected item in the list
     * @return result -  index of the selected item in the list
     */
    public int showDialog() {
        setVisible(true);
        return result;
    }
}