package de.renew.engine.simulator;

import org.junit.Before;
import org.junit.Test;

import de.renew.util.Semaphor;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.mockito.Mockito.atLeast;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;


/**
 * @author Felix Ortmann
 *
 * Testcase for the {@link BlockingSimulationRunnable}.
 *
 */
public class TestBlockingSimulationRunnable {
    private Runnable _runnable;
    private BlockingSimulationRunnable _block1;
    private BlockingSimulationRunnable _block2;
    private BlockingSimulationRunnable _block3;
    private Semaphor _lock;
    private Thread _thread;

    //since we have no return-statements in the runnables, we will change a String
    private String _changeableTestString1;
    private String _changeableTestString2;
    private String _changeableTestString3;
    private SimulationThreadPool _pool;

    private static final String TEST_THREAD_NAME = "test";
    private static final String TEST_STRING_1 = "1";
    private static final String TEST_STRING_2 = "2";
    private static final String TEST_STRING_3 = "3";

    /**
     * SetUp for testing the BlockingSimulationRunnable
     * This method is invoked before the execution of each single test method
     */
    @Before
    public void setUp() {
        _changeableTestString1 = null;
        _changeableTestString2 = null;
        _changeableTestString3 = null;

        _lock = mock(Semaphor.class);
        _thread = Thread.currentThread();

        _pool = SimulationThreadPool.getNew();

        _runnable = () -> _changeableTestString1 = TEST_STRING_1;

        Runnable innerRun1 = () -> _changeableTestString2 = TEST_STRING_2;
        Runnable innerRun2 = () -> _changeableTestString3 = TEST_STRING_3;

        //initialize the blockings under test
        _block1 = new BlockingSimulationRunnable(_runnable, _lock, _thread);
        _block2 = new BlockingSimulationRunnable(innerRun1, _lock, _thread);
        _block3 = new BlockingSimulationRunnable(innerRun2, _lock, null);

    }


    /**
     * Test method for {@link de.renew.engine.simulator.BlockingSimulationRunnable#getAncestor()}.
     */
    @Test
    public final void testGetAncestor() {
        assertEquals(_thread, _block1.getAncestor());
        assertEquals(_thread, _block2.getAncestor());
        assertNull(_block3.getAncestor());
    }

    /**
     * Test method for {@link de.renew.engine.simulator.BlockingSimulationRunnable#run()}.
     */
    @Test
    public final synchronized void testRun() {
        _pool.execute(_block1);
        _pool.execute(_block2);
        _pool.execute(_block3);

        try {
            System.out.println("Waiting a bit for the Runnable-Execution to finish.");
            wait(1500);
        } catch (InterruptedException e) {
            // we want the pool to have enough time to execute each runnable concurrently
            e.printStackTrace();
        }

        assertEquals(TEST_STRING_1, _changeableTestString1);
        assertEquals(TEST_STRING_2, _changeableTestString2);
        assertEquals(TEST_STRING_3, _changeableTestString3);

        //test the unlocking behavior of the run-methods final statement
        verify(_lock, atLeast(3)).V();
    }

    /**
     * Test method for {@link de.renew.engine.simulator.BlockingSimulationRunnable#abort(de.renew.engine.simulator.SimulationThread)}
     */
    @Test
    public final void testAbort() {
        _block1.abort(null);
        assertNull(_block1.getAncestor());
        verify(_lock, atLeastOnce()).V();

        SimulationThread simThread =
            new SimulationThread(_thread.getThreadGroup(), _runnable, TEST_THREAD_NAME, 5);

        _block2.abort(simThread);
        assertNull(_block2.getAncestor());
        assertNull(simThread.getAncestor());
        verify(_lock, atLeastOnce()).V();
    }
}