package de.renew.net.inscription;

import java.util.Collection;
import java.util.Vector;

import de.renew.engine.common.DownlinkOccurrence;
import de.renew.engine.searcher.Occurrence;
import de.renew.engine.searcher.Searcher;
import de.renew.expression.Expression;
import de.renew.expression.VariableMapper;
import de.renew.net.NetInstance;
import de.renew.net.Transition;
import de.renew.net.TransitionInscription;

/**
 * A {@code DownlinkInscription} is used to indicate that the corresponding transition is the downlink of a certain
 * synchronous channel. For example, a {@code DownlinkInscription} is created for a transition inscribed with
 * "this:lookup(x,y)". Then the transition is an initiating transition of the synchronous channel "lookup", whose
 * uplink is located in the same net instance, with the parameters x and y.
 */
public class DownlinkInscription implements TransitionInscription {
    /**
     * The name of the synchronous channel.
     */
    public String name;

    /**
     * The {@code Expression} evaluating to variables that will be unified with the synchronous
     * channel's parameters.
     */
    public Expression params;

    /**
     * The {@code Expression} evaluating to a variable containing the reference to a net instance
     * that contains the channel's uplink.
     */
    public Expression callee;

    /**
     * {@code true} if the downlink is optional.
     */
    boolean isOptional;

    /**
     * The {@code Transition} that contains the {@code DownlinkInscription}.
     */
    public Transition transition;

    /**
     * Constructs a new {@code DownlinkInscription} with the given parameters.
     *
     * @param name the name of the synchronous channel
     * @param params the {@code Expression} representing the parameters of the synchronous channel
     * @param callee the {@code Expression} representing the net instance containing the channel's uplink
     * @param isOptional {@code true} if the downlink is optional
     * @param transition the {@code Transition} that contains the {@code DownlinkInscription}
     */
    public DownlinkInscription(
        String name, Expression params, Expression callee, boolean isOptional,
        Transition transition)
    {
        this.name = name;
        this.params = params;
        this.callee = callee;
        this.isOptional = isOptional;
        this.transition = transition;
    }

    // Refactoring
    /**
     * The start index of the downlink line.
     */
    public int downlinkBeginLine;

    /**
     * The start index of the downlink column.
     */
    public int downlinkBeginColumn;

    /**
     * The end index of the downlink line.
     */
    public int downlinkEndLine;

    /**
     * The end index of the downlink column.
     */
    public int downlinkEndColumn;

    /**
     * The start index of the name line.
     */
    public int nameBeginLine;

    /**
     * The start index of the name column.
     */
    public int nameBeginColumn;

    /**
     * The end index of the name column.
     */
    public int nameEndColumn;

    /**
     * The end index of the name line.
     */
    public int nameEndLine;

    @Override
    public java.util.Collection<Occurrence> makeOccurrences(
        VariableMapper mapper, NetInstance netInstance, Searcher searcher)
    {
        Collection<Occurrence> coll = new Vector<Occurrence>();
        coll.add(
            new DownlinkOccurrence(
                params, callee, name, isOptional, mapper, netInstance.getInstance(transition)));
        return coll;
    }
}