package de.renew.net.arc;

import java.util.Vector;

import de.renew.engine.common.StepIdentifier;
import de.renew.engine.searcher.EarlyExecutable;
import de.renew.engine.searchqueue.SearchQueue;
import de.renew.net.SimulatablePlaceInstance;
import de.renew.net.TransitionInstance;
import de.renew.unify.Impossible;

/**
 * The executable belonging to {@link FlexibleArcOccurrence} instances that are input arcs.
 */
class FlexibleInArcExecutable implements EarlyExecutable {
    /** The place instance the arc takes tokens from. */
    SimulatablePlaceInstance pInstance;
    /** The transition instance that the arc is connected to. */
    TransitionInstance tInstance;
    /** The tokens removed from {@link #pInstance} by the flexible arc. */
    Vector<Object> tokens;
    /**
     * The flexible arc whose {@code FlexibleArcOccurrence} in a {@link de.renew.net.NetInstance NetInstance}
     * this {@code FlexibleInArcExecutable} belongs to.
     */
    FlexibleArc arc;
    /** The amount of tokens that have been removed from {@link #pInstance}. */
    int numRemoved;
    /** The times at which tokens have been removed from {@link #pInstance}. */
    Vector<Double> removedTimes;

    /**
     * Constructs a new {@code FlexibleInArcExecutable} based on the instances of the
     * place and transition the arc connects, the tokens that should be moved during execution, and the arc itself.
     *
     * @param placeInstance the place instance that tokens will be taken from
     * @param tInstance the transition instance that is connected to the arc
     * @param tokens the tokens that will be removed
     * @param arc the arc whose occurrence in a net instance causes the executable to be created
     */
    FlexibleInArcExecutable(
        SimulatablePlaceInstance placeInstance, TransitionInstance tInstance, Vector<Object> tokens,
        FlexibleArc arc)
    {
        this.pInstance = placeInstance;
        this.tInstance = tInstance;
        this.tokens = tokens;
        this.arc = arc;
        numRemoved = 0;
        removedTimes = new Vector<Double>();
    }

    @Override
    public long lockPriority() {
        return pInstance.lockOrder;
    }

    @Override
    public int phase() {
        return INPUT;
    }

    /**
     * Locks the {@link de.renew.net.PlaceInstance#lock PlaceInstance} associated with this arc.
     **/
    @Override
    public void lock() {
        pInstance.lock.lock();
    }

    @Override
    public void verify(StepIdentifier stepIdentifier) throws Impossible {
        try {
            for (int i = 0; i < tokens.size(); i++) {
                Object token = tokens.elementAt(i);
                double removedTime = pInstance.removeToken(token, 0);
                removedTimes.addElement(Double.valueOf(removedTime));
                numRemoved++;
            }
        } catch (Exception e) {
            // Undo all reservations.
            rollback();

            throw new Impossible();
        }
    }

    @Override
    public void execute(StepIdentifier stepIdentifier) {
        for (int i = 0; i < numRemoved; i++) {
            Object token = tokens.elementAt(i);
            if (arc.arcType == FlexibleArc.fastBoth) {
                if (arc.trace) {
                    InputArcExecutable
                        .traceInArc(stepIdentifier, true, token, pInstance, tInstance); //NOTICEsignature
                }
                pInstance.insertToken(token, SearchQueue.getTime());
            } else {
                if (arc.trace) {
                    InputArcExecutable
                        .traceInArc(stepIdentifier, false, token, pInstance, tInstance); //NOTICEsignature
                }
            }
        }
    }

    @Override
    public void rollback() {
        // We have to undo the previous removals. We cannot do this
        // without notifying the observers and listeners, because
        // it was not done silently. However, the database must not log
        // this modification.
        for (int i = 0; i < numRemoved; i++) {
            pInstance.internallyInsertToken(
                tokens.elementAt(i), removedTimes.elementAt(i).doubleValue(), false);
        }
    }

    /**
     * Unlocks the {@link de.renew.net.PlaceInstance#lock PlaceInstance} associated with this arc.
     **/
    @Override
    public void unlock() {
        pInstance.lock.unlock();
    }
}