package de.renew.engine.searcher;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import de.renew.engine.simulator.SimulationThreadPool;
import de.renew.unify.CalculationChecker;
import de.renew.unify.Impossible;
import de.renew.unify.StateRecorder;
import de.renew.unify.Variable;

public class Searcher implements ISearcher {
    public static final org.apache.log4j.Logger logger =
        org.apache.log4j.Logger.getLogger(Searcher.class);

    /**
     * The object that is informed about every
     * successfully detected binding.
     **/
    private Finder finder;

    /**
     * the set of current unprocessed {@link Binder}s
     **/
    private final Set<Binder> binders;

    /**
     * the set of {@link Occurrence} object (normally
     * transition occurrences) that participate
     * in this search and in the subsequent execution.
     **/
    private final Set<Occurrence> occurrences;

    /**
     * the object which is informed about
     **/
    private Triggerable primaryTriggerable;
    private double earliestTime;

    /**
     * @deprecated Use {@link Searcher#getCalculationChecker()} instead.
     * Internal code may still use this field.
     */
    @Deprecated
    public CalculationChecker calcChecker;

    /**
     * @deprecated Use {@link Searcher#getStateRecorder()} instead.
     * Internal code may still use this field.
     */
    @Deprecated
    public StateRecorder recorder;
    private final Map<String, DeltaSet> deltaSets;

    /**
     * Construct a cleanly initialized instance.
     */
    public Searcher() {
        this(new HashSet<>(), new HashMap<>());
    }

    /**
     * Constructor setting the primary triggerable and the finder to the given values.
     * <p>
     * This constructor is used by the Distribute plug-in
     * ({@literal @link de.renew.distribute.DistributePlugin})
     * to search a local target net instance
     * for another Renew simulation.
     * If another local target is already involved in the distributed search,
     * {@link #Searcher(Searcher, Finder)} is used instead.
     * <p>
     * This constructor is called in
     * {@literal @link DistributeSearcher#DistributeSearcher(SearchDistributor, Triggerable, Finder)}
     * to prepare a call to {@literal @link DistributeSearcher#searchFirstCall}.
     * That sequence is used in {@literal @link SearcherAccessorImpl#searchCall}.
     */
    protected Searcher(Triggerable primaryTriggerable, Finder finder) {
        this();
        this.primaryTriggerable = primaryTriggerable;
        this.finder = finder;
    }

    /**
     * Constructor initializing this searcher to a state similar to {@code parent}'s.
     * This searcher's state will not be completely equal.
     * Especially {@link #calcChecker}, {@link #recorder}
     * and the internal set of {@link Binder} objects are not copied.
     * <p>
     * This constructor is used by the Distribute plug-in
     * ({@literal @link de.renew.distribute.DistributePlugin})
     * to search an additional local target net instance
     * for another Renew simulation.
     * It is called in {@literal @link DistributeSearcher#searchAdditionalCall}
     * and this in turn is called in {@literal @link SearcherAccessorImpl#searchCall}.
     */
    protected Searcher(Searcher parent, Finder finder) {
        this(parent.occurrences, parent.deltaSets);
        this.primaryTriggerable = parent.primaryTriggerable;
        this.finder = finder;
    }

    private Searcher(Set<Occurrence> occurrences, Map<String, DeltaSet> deltaSets) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        binders = new HashSet<>();
        this.occurrences = occurrences;

        calcChecker = new CalculationChecker();
        recorder = new StateRecorder();

        this.deltaSets = deltaSets;
    }

    @Override
    public CalculationChecker getCalculationChecker() {
        return calcChecker;
    }

    @Override
    public StateRecorder getStateRecorder() {
        return recorder;
    }

    @Override
    public boolean isCompleted() {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        return finder.isCompleted();
    }

    private void setPrimaryTriggerable(Triggerable triggerable) {
        // Clear all triggers previously associated to the triggerable.
        if (triggerable != null) {
            triggerable.triggers().clear();
        }

        primaryTriggerable = triggerable;
    }

    protected Triggerable getPrimaryTriggerable() {
        return primaryTriggerable;
    }

    @Override
    public void insertTriggerable(TriggerableCollection triggerables) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        // If there is a primary triggerable, we register it
        // at the given collection of triggerables.
        if (primaryTriggerable != null) {
            triggerables.include(primaryTriggerable);
        }
    }

    @Override
    public void addOccurrence(Occurrence occurrence) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        occurrences.add(occurrence);
    }

    @Override
    public void removeOccurrence(Occurrence occurrence) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        occurrences.remove(occurrence);
    }

    @Override
    public Collection<Occurrence> getOccurrences() {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        return occurrences;
    }

    @Override
    public DeltaSet getDeltaSet(DeltaSetFactory factory) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        String category = factory.getCategory();
        DeltaSet result = deltaSets.get(category);
        if (result == null) {
            result = factory.createDeltaSet();
            deltaSets.put(category, result);
        }
        return result;
    }

    @Override
    public double getEarliestTime() {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        return earliestTime;
    }

    /**
     * Remember the earliest possible time for
     * the currently found binding to be enacted.
     * This value can later on be queried by
     * {@link #getEarliestTime}.
     *
     * @param time the time
     **/
    private void setEarliestTime(double time) {
        earliestTime = time;
    }

    /**
     * Find the binder with the minimum binding badness
     * Return null, if no binder wants to try.
     **/
    private Binder selectBestBinder() {
        Binder bestBinder = null;
        int bestBadness = BindingBadness.max;
        Iterator<Binder> enumeration = binders.iterator();
        while (enumeration.hasNext() && !isCompleted()) {
            Binder binder = enumeration.next();
            int badness = binder.bindingBadness(this);
            if (badness < bestBadness) {
                bestBinder = binder;
                bestBadness = badness;
            }
        }
        return bestBinder;
    }

    @Override
    public void search() {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        int checkpoint = recorder.checkpoint();
        if (!binders.isEmpty()) {
            Binder binder = selectBestBinder();
            if (binder != null && !isCompleted()) {
                removeBinder(binder);
                binder.bind(this);
                addBinder(binder);
            }
        } else {
            if (calcChecker.isConsistent()) {
                // Make sure not to fire before all tokens are available.
                double time = 0;
                for (DeltaSet deltaSet : deltaSets.values()) {
                    time = Math.max(time, deltaSet.computeEarliestTime());
                }
                setEarliestTime(time);


                // Notify the finder, even if the binding is not yet
                // activated. The finder might want to store the
                // binding for a later time or find out the earliest possible
                // binding.
                finder.found(this);
            }
        }
        recorder.restore(checkpoint);
    }

    @Override
    public void search(Occurrence occurrence) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        int checkpoint = recorder.checkpoint();
        try {
            Collection<Binder> binders = occurrence.makeBinders(this);
            addOccurrence(occurrence);
            addBinders(binders);
            search();
            removeBinders(binders);
            removeOccurrence(occurrence);
        } catch (Impossible e) {
            // When getting the binders, an exception was thrown.
            // The occurrence cannot be enabled.
        } finally {
            recorder.restore(checkpoint);
        }
    }

    @Override
    public void addBinder(Binder binder) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        binders.add(binder);
    }

    @Override
    public void removeBinder(Binder binder) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        binders.remove(binder);
    }

    @Override
    public void addBinders(Collection<Binder> binders) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        this.binders.addAll(binders);
    }

    @Override
    public void removeBinders(Collection<Binder> binders) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        this.binders.removeAll(binders);
    }

    /**
     * Make sure to clean up before a search.
     **/
    private void startSearch() {
        // Make sure that the current state recorder does not waste
        // space with now obsolete bindings.
        recorder.restore();


        // Ensure that the temporary containers are initially empty.
        // If they are not, something went wrong previously,
        // so we report en error.
        if (!occurrences.isEmpty() || !binders.isEmpty()) {
            throw new RuntimeException(
                "Searcher was not in idle state " + "at the start of a search.");
        }
    }

    @Override
    public void searchAndRecover(Finder finder, Searchable searchable, Triggerable triggerable) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        // Prepare a new search that starts from scratch.
        startSearch();


        // Remember the finder.
        this.finder = finder;

        // Remember the triggerable that is interested in changes
        // of this search result.
        setPrimaryTriggerable(triggerable);


        // Unless we find any further restriction,
        // the next firing can happen immediately.
        earliestTime = 0;

        try {
            // Is this searchable object activated?
            searchable.startSearch(this);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);


            // We reinsert this searchable only when a binding was found,
            // because otherwise there is no real chance that we
            // could have more luck next time.
        }


        // Reset state to allow garbage collection.
        binders.clear();
        occurrences.clear();

        calcChecker.reset();
        recorder.restore();
        deltaSets.clear();

        setPrimaryTriggerable(null);

        this.finder = null;
    }

    @Override
    public void initiatedSearch(
        ChannelTarget channelTarget, String name, Variable params, boolean isOptional,
        Finder finder, Triggerable triggerable)
    {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        // Prepare a new search that starts from scratch.
        startSearch();


        // Remember the finder.
        this.finder = finder;


        // Possible register for state change notifications.
        setPrimaryTriggerable(triggerable);

        try {
            // We want to find a transition within the specified
            // net instance that can fire. This method is
            // usually called to generate the very first tokens
            // in a net. In a sense, this resembles the main(args)
            // call of the runtime environment.
            //
            // If the argument isOptional is true, the invoked net
            // need not provide an appropriate channel at all. But if it does,
            // the synchronisation must succeed.
            Variable targetVariable = new Variable(channelTarget, recorder);
            Binder initialBinder = new ChannelBinder(targetVariable, name, params, isOptional);
            addBinder(initialBinder);
            search();
            removeBinder(initialBinder);
            recorder.restore();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);


            // We reinsert this searchable only when a binding was found,
            // because otherwise there is no real chance that we
            // could have more luck next time.
        }


        // Reset state to allow garbage collection.
        binders.clear();
        occurrences.clear();

        calcChecker.reset();
        recorder.restore();
        deltaSets.clear();

        setPrimaryTriggerable(null);
        this.finder = null;
    }
}