package simulator.cross.module;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import de.renew.application.SimulationRunningException;
import de.renew.engine.simulator.SimulationThreadPool;
import de.renew.net.Net;
import de.renew.net.NetElementID;
import de.renew.net.Place;
import de.renew.util.RenewObjectInputStream;
import de.renew.util.RenewObjectOutputStream;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;

/**
 * This class is a test class for the interaction between the classes {@link Place} from the module
 * de.renew.Simulator and {@link RenewObjectOutputStream} and {@link RenewObjectInputStream}
 * from the module de.renew.util.
 */
public class PlaceTest {
    String path = "src" + File.separator + "integTest" + File.separator + "resources"
        + File.separator + "test.ser";
    File testFile = new File(path);
    RenewObjectOutputStream renewObjectOutputStream;
    RenewObjectInputStream renewObjectInputStream;
    Place place;
    Place readPlace;
    boolean done;

    /**
     * This method should be executed before each test method.
     * @throws IOException
     */
    @BeforeEach
    void setUp() throws IOException {
        if (testFile.exists()) {
            testFile.delete();
        }
        done = false;
        if (!testFile.exists()) {
            testFile.createNewFile();
        }
        FileOutputStream fileOutputStream = new FileOutputStream(testFile);
        FileInputStream fileInputStream = new FileInputStream(testFile);

        renewObjectOutputStream = new RenewObjectOutputStream(fileOutputStream);
        renewObjectInputStream = new RenewObjectInputStream(fileInputStream);
    }

    /**
     * This method should be executed after each test method.
     * @throws IOException
     */
    @AfterEach
    void teardown() {
        done = false;
    }

    /**
     * Tests the methods {@link RenewObjectOutputStream#writeObject(Object)}
     * and {@link RenewObjectInputStream#readObject()}
     * It creates a place, serializes it, deserializes it and compares the two places.
     * @throws IOException
     * @throws InterruptedException
     */
    @Test
    void testWriteAndReadObjectPlace() throws IOException, InterruptedException {
        Runnable runnable = new Runnable() {
            @Override
            public void run() {
                try {
                    Net net = new Net();
                    NetElementID netElementID = new NetElementID();
                    String name = "Test";
                    place = new Place(net, name, netElementID);
                    renewObjectOutputStream.writeObject(place);

                    readPlace = (Place) renewObjectInputStream.readObject();
                    done = true;

                } catch (SimulationRunningException | IOException | ClassNotFoundException e) {
                    SimulationThreadPool.discardNew();
                    try {
                        throw e;
                    } catch (IOException | ClassNotFoundException ex) {
                        ex.printStackTrace();
                    }
                }

            }
        };
        runBySimulationThread(runnable);

        assertNotNull(readPlace);
        assertEquals(place.getName(), readPlace.getName());
        assertEquals(place.getID(), readPlace.getID());
    }

    /**
     * Runs a runnable by a simulation thread.
     * @param runnable
     * @throws InterruptedException
     */
    private void runBySimulationThread(Runnable runnable) throws InterruptedException {
        done = false;
        SimulationThreadPool.getNew();
        SimulationThreadPool simulationThreadPool = SimulationThreadPool.getSimulationThreadPool();
        Thread simulationThread = simulationThreadPool.getThreadFactory().newThread(runnable);
        simulationThread.start();
        while (!done) {
            System.out.println("Simulation thread is running");
            Thread.sleep(2000);
        }
    }
}