package de.renew.navigator.io;

import java.io.File;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collection;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;

import CH.ifa.draw.io.CombinationFileFilter;
import de.renew.io.RNWFileFilter;
import de.renew.navigator.models.Directory;
import de.renew.navigator.models.Leaf;
import de.renew.navigator.models.TreeElement;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertSame;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;


/**
 * @author Konstantin Simon Maria Moellers
 * @version 2015-10-09
 */
public class FilesystemIOLoaderTest {
    private FileFilterBuilder _builder;
    private FilesystemIOLoader _loader;
    private CombinationFileFilter _ff;
    private File _fixtures;

    private static TreeElement findChildByName(Collection<TreeElement> elements, String name) {
        for (TreeElement element : elements) {
            if (element.getName().equals(name)) {
                return element;
            }
        }

        return null;
    }

    @BeforeEach
    public void setUp() throws Exception {
        _builder = mock(FileFilterBuilderImpl.class);
        _loader = new FilesystemIOLoader(_builder);

        // Create file filter.
        _ff = new CombinationFileFilter("Stub");
        _ff.add(new RNWFileFilter());
        when(_builder.buildFileFilter()).thenReturn(_ff);

        // Load fixtures.
        final URL resource = getClass().getResource(getBaseDirectory());
        _fixtures = new File(resource.toURI());

    }

    private String getBaseDirectory() {
        return "/" + Paths.get("de", "renew", "navigator", "io", "fixtures");
    }

    @Test
    public void testLoadFile() throws Exception {
        ProgressListener listener = mock(ProgressListener.class);
        Path path = Paths.get(getBaseDirectory(), "testfile.txt");
        final File testFile = new File(path.toString());

        final TreeElement result = _loader.loadPath(testFile, listener);

        verifyNoMoreInteractions(_builder);
        verifyNoMoreInteractions(listener);

        // Assert content of results.
        assertEquals(result.getFile(), testFile);
        assertEquals(result.getName(), testFile.getName());
        assertNull(result.getParent());
        assertTrue(result instanceof Leaf);
    }

    @Test
    public void testLoadDirectory() throws Exception {
        ProgressListener listener = mock(ProgressListener.class);

        final TreeElement result = _loader.loadPath(_fixtures, listener);

        verify(_builder).buildFileFilter();

        // Assert content of results.
        assertEquals(result.getFile(), _fixtures);
        assertEquals(result.getName(), _fixtures.getName());
        assertNull(result.getParent());
        assertTrue(result instanceof Directory);

        final Directory dir = (Directory) result;
        assertEquals(dir.getChildren().size(), _fixtures.listFiles(_ff).length);
        assertEquals(6, dir.getChildren().size());
        assertTrue(dir.isOpened());
        assertNull(dir.getType());

        for (TreeElement child : dir.getChildren()) {
            assertSame(child.getParent(), dir);
        }
    }

    @Test
    @Disabled
    public void testRenameFile() {
        ProgressListener listener = mock(ProgressListener.class);
        String dir = _fixtures.getAbsolutePath() + FileSystems.getDefault().getSeparator();

        final File fig1 = new File(dir + "fig1.rnw");
        final File fig2 = new File(dir + "Fig1.rnw");
        final File myfig1 = new File(dir + "myfig1.rnw");


        assertTrue(fig1.isFile());
        try {
            final Directory result = (Directory) _loader.loadPath(_fixtures, listener);
            assertEquals(6, result.getChildren().size());
            assertNotNull(findChildByName(result.getChildren(), "Fig1.rnw"));
            assertNull(findChildByName(result.getChildren(), "myfig1.rnw"));
            assertNull(findChildByName(result.getChildren(), "fig1.rnw"));

            if (!fig1.renameTo(fig2)) {
                fail("Could not rename the file 'fig1.rnw'. Please check your filesystem!");
            }

            _loader.refreshPath(result, _fixtures, listener);
            assertEquals(6, result.getChildren().size());
            assertNull(findChildByName(result.getChildren(), "fig1.rnw"));
            assertNull(findChildByName(result.getChildren(), "myfig1.rnw"));
            assertNotNull(findChildByName(result.getChildren(), "Fig1.rnw"));

            if (!fig2.renameTo(myfig1)) {
                fail("Could not rename the file 'Fig1.rnw'. Please check your filesystem!");
            }

            _loader.refreshPath(result, _fixtures, listener);
            assertEquals(6, result.getChildren().size());
            assertNull(findChildByName(result.getChildren(), "fig1.rnw"));
            assertNull(findChildByName(result.getChildren(), "Fig1.rnw"));
            assertNotNull(findChildByName(result.getChildren(), "myfig1.rnw"));

        } finally {
            // Retrieve initial state.
            if (!myfig1.renameTo(fig1)) {
                fail("Could not rename the file 'myfig1.rnw' back to 'fig1.rnw'??");
            }
        }
    }
}