package de.renew.plugin.locate;

import java.net.URL;
import java.util.Collection;

import de.renew.plugin.PluginProperties;


/**
 * This interface abstracts the process of looking for plugins
 * (in the file system, remote).
 * It retrieves a Collection which must contain PluginProperties objects.
 * These must be present previous to actually loading the plugins themselves
 * to enable the dependency check.
 *
 * @author Joern Schumacher
 * @author Michael Duvigneau
 */
public interface PluginLocationFinder {

    /**
     * Returns all plugins this finder knows about.
     *
     * @return a <code>Collection</code> of {@link PluginProperties}.
     **/
    Collection<PluginProperties> findPluginLocations();

    /**
     * Checks whether the given <code>URL</code> describes a
     * plugin. If this is the case, the method returns the
     * corresponding <code>PluginProperties</code> object.
     * Otherwise, it returns <code>null</code>.
     *
     * @param url  the <code>URL</code> to check
     * @return a <code>PluginProperties</code> value
     **/
    PluginProperties checkPluginLocation(URL url);

    /**
     * Finds the PluginProperties corresponding to provide entry.
     *
     * @param provides The name of provided dependency we want to find
     * @return the PluginProperties corresponding to provide entry. It can be null
     */
    PluginProperties findPluginByProvides(String provides);

    /**
     * Finds the PluginProperty corresponding to the plugin name.
     *
     * @param pluginName the name of the plugin
     * @return the PluginProperty corresponding to the plugin name
     */
    PluginProperties findPluginByName(String pluginName);

    /**
     * Returns all plugin location
     *
     * @return Collection of plugin locations as URLs
     */
    Collection<URL> getLocations();
}