package de.renew.plugin.di;

import de.renew.plugin.IPlugin;
import de.renew.plugin.PluginProperties;


/**
 * {@code DIPlugin} represents the class for the DI plugin.
 * @author Konstantin Simon Maria Moellers
 * @version 2015-10-10
 */
public abstract class DIPlugin implements IPlugin {
    PluginProperties _properties;

    /**
     * @return  The default implementation obtains the name of the plugin
     *          from the associated {@link PluginProperties} object.
     */
    @Override
    final public String getName() {
        return _properties.getName();
    }

    /**
     * Returns plugin alias taken from plugin properties
     * @return  The default implementation obtains the alias of the plugin
     *          from the associated {@link PluginProperties} object.
     */
    @Override
    final public String getAlias() {
        return _properties.getProperty("alias");
    }

    /**
     * The version is now given in etc/plugin.cfg and will be displayed also when "info nc"
     * is called on the command prompt.
     * @return  The Version information.
     */
    @Override
    public String getLongVersion() {
        String result = "";
        PluginProperties props = getProperties();
        if (props.containsKey("version")) {
            String versionText = props.getProperty("versionText");
            if (versionText == null) {
                versionText = "Version: ";
            }

            String versionDate = props.getProperty("versionDate");
            if (versionDate == null) {
                versionDate = "no date set";
            }
            result = versionText + " " + getVersion() + "! Date: " + versionDate;
        }
        return result;
    }

    /**
     * The version is now given in etc/plugin.cfg and will be displayed also when "info nc"
     * is called on the command prompt.
     * @return  The Version information.
     */
    @Override
    final public String getVersion() {
        if (_properties.containsKey("version")) {
            return _properties.getProperty("version");
        }

        return "0.0.1";
    }

    /**
     * Builds a String representation of the plugin name with plugin as prefix
     * @return  The default implementation returns the name of the plugin,
     *          prepended by the word <code>plugin</code>.
     */
    @Override
    public String toString() {
        return "plugin " + getName();
    }

    /**
     * Sets the plugin properties.
     *
     * @param properties The properties to set for this plugin
     */
    final public void setProperties(PluginProperties properties) {
        this._properties = properties;
    }

    @Override
    final public PluginProperties getProperties() {
        return _properties;
    }

    @Override
    public void init() {}

    @Override
    public boolean cleanup() {
        return true;
    }

    @Override
    public boolean canShutDown() {
        return true;
    }

    @Override
    public void startUpComplete() {}

    /**
     * Can provide additional functionality for a plugin after Renew is started.
     * <p>
     * This method calls {@link #startUpComplete()} no matter how the parameter is set.
     * Overwriting subclasses (Plugins) can react differently depending on
     * the time they were loaded by overwriting this method and reacting differently on the given parameter.
     *
     * @param initialStart true, when the plugin is loaded during the initial Renew start, false otherwise.
     */
    @Override
    public void startUpComplete(boolean initialStart) {
        startUpComplete();
    }
}