package de.renew.gui.configure;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionListener;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;


/**
 * A configuration dialog for the Renew simulation.
 * The dialog is managed by {@link ConfigureSimulationController}.
 *
 * @author Michael Duvigneau
 **/
class ConfigureSimulationDialog extends JDialog {
    private JTabbedPane _optionsTabbedPane;
    private JButton _applyButton;
    private JButton _updateButton;
    private JButton _updateSimButton;
    private JButton _okButton;
    private JButton _closeButton;

    /**
     * Creates the new dialog and its components. The given
     * <code>initalTabs</code> are added to the options tabbed
     * pane.
     * @param parent      the frame to which this dialog belongs.
     * @param initialTabs the option panels to add to the dialog
     *                    before it gets layouted.
     **/
    ConfigureSimulationDialog(JFrame parent, Component[] initialTabs) {
        super(parent, false);
        initComponents();
        for (Component initialTab : initialTabs) {
            _optionsTabbedPane.add(initialTab);
        }
        pack();
    }

    /**
     * Creates and initialises the dialog's fixed components.
     **/
    private void initComponents() {
        _optionsTabbedPane = new JTabbedPane();
        JPanel buttonPanel = new JPanel();

        setName("Configure Simulation");
        setTitle("Configure Simulation");

        JLabel noteLabel = new JLabel();
        noteLabel.setText("Settings will not take effect until a new simulation is set up.");
        getContentPane().add(noteLabel, BorderLayout.NORTH);

        getContentPane().add(_optionsTabbedPane, BorderLayout.CENTER);

        _applyButton = new JButton();
        _applyButton.setText("Apply");
        _applyButton.setToolTipText(
            "Confirms these settings for the next simulation setup and closes the dialog.");
        buttonPanel.add(_applyButton);

        _updateButton = new JButton();
        _updateButton.setToolTipText(
            "Copies the current settings of the simulator plugin into this dialog.");
        _updateButton.setText("Update");
        buttonPanel.add(_updateButton);

        _updateSimButton = new JButton();
        _updateSimButton
            .setToolTipText("Copies the settings of the current simulation into this dialog.");
        _updateSimButton.setText("Update from Simulation");
        buttonPanel.add(_updateSimButton);

        _okButton = new JButton();
        _okButton.setText("OK");
        _okButton.setToolTipText(
            "Confirms these settings for the next simulation setup and closes the dialog.");
        buttonPanel.add(_okButton);

        _closeButton = new JButton();
        _closeButton.setToolTipText("Closes the dialog without changing the settings.");
        _closeButton.setText("Close");
        buttonPanel.add(_closeButton);

        getContentPane().add(buttonPanel, BorderLayout.SOUTH);
    }

    /**
     * Adds an ActionListener to the close button of the dialog. The provided
     * ActionListener will be notified when the close button is triggered.
     *
     * @param listener the ActionListener to attach to the close button
     */
    void addCloseActionListener(ActionListener listener) {
        this._closeButton.addActionListener(listener);
    }

    /**
     * Adds an ActionListener to the OK button of the dialog.
     * The provided ActionListener will be notified when the OK button is triggered.
     *
     * @param listener the ActionListener to attach to the OK button
     */
    void addOkActionListener(ActionListener listener) {
        this._okButton.addActionListener(listener);
    }

    /**
     * Adds the specified ActionListener to the "Apply" button in the dialog.
     * This allows custom actions to be executed when the "Apply" button is pressed.
     *
     * @param listener the ActionListener to be added to the "Apply" button.
     */
    void addApplyActionListener(ActionListener listener) {
        this._applyButton.addActionListener(listener);
    }

    /**
     * Adds an ActionListener to the "Update from Simulation" button. The provided
     * ActionListener will be notified when the button is triggered.
     *
     * @param listener the ActionListener to attach to the "Update Simulation" button
     */
    void addUpdateSimulationActionListener(ActionListener listener) {
        this._updateSimButton.addActionListener(listener);
    }

    /**
     * Adds an ActionListener to the "Update" button in the dialog.
     * The provided ActionListener will be notified when the button is triggered.
     *
     * @param listener the ActionListener to attach to the "Update" button
     */
    void addUpdateActionListener(ActionListener listener) {
        this._updateButton.addActionListener(listener);
    }

    /**
     * Adds the given Swing component as a new tab to the dialog.
     * @param tab  the options panel to add to the dialog.
     **/
    void addOptionsTab(Component tab) {
        _optionsTabbedPane.add(tab);
    }

    /**
     * Removes the given Swing component from this dialog's
     * tabbed pane.
     * @param tab  the options panel to remove from the dialog.
     **/
    void removeOptionsTab(Component tab) {
        _optionsTabbedPane.remove(tab);
    }

    /**
     * Removes all tabs from this dialog's tabbed pane.
     **/
    void removeAllOptionTabs() {
        _optionsTabbedPane.removeAll();
    }
}
