/*
 * Created on 27.01.2004
 *
 */
package de.renew.gui;

import java.awt.Component;
import java.util.LinkedHashSet;
import java.util.Vector;
import javax.swing.AbstractButton;
import javax.swing.Icon;
import javax.swing.JPanel;
import javax.swing.JToolBar;

import CH.ifa.draw.framework.Tool;
import CH.ifa.draw.framework.Toolbar;
import CH.ifa.draw.standard.ToolButton;
import CH.ifa.draw.util.Palette;
import CH.ifa.draw.util.PaletteListener;
import de.renew.windowmanagement.Workbench;


/**
 * This class represents the palettes provided by the Gui plugin.
 * It is used to receive delegated PaletteHolder calls.
 *
 * @author J&ouml;rn Schumacher
 */
public class GuiPalettes implements PaletteHolder {
    private static final org.apache.log4j.Logger logger =
        org.apache.log4j.Logger.getLogger(GuiPalettes.class);
    private final TextFigureToolButtonRegistry textFigureToolButtonRegistry;

    /**
     * A list of all palettes that have been registered.
     **/
    private final Vector<Palette> _palettes = new Vector<>();
    private final GuiPlugin _plugin;
    private final Workbench _workbench;

    /**
     * The PaletteListener that receives Palette Events from ToolButtons and such.
     * If a Gui is currently open, it delegates the event there.
     **/
    final GuiEventDelegator _eventDelegator = new GuiEventDelegator();

    public GuiPalettes(Workbench workbench, GuiPlugin plugin) {
        _plugin = plugin;
        _workbench = workbench;
        this.textFigureToolButtonRegistry = new TextFigureToolButtonRegistry();
    }

    void notifyGuiOpen() {
        for (Palette toAdd : _palettes) {
            internalAddPalette(toAdd);
        }
    }

    /*
     * @see de.renew.gui.PaletteHolder#addPalette(CH.ifa.draw.util.Palette)
     */
    @Override
    public void addPalette(Palette toAdd) {
        _palettes.add(toAdd);
        internalAddPalette(toAdd);
    }

    private void internalAddPalette(Palette toAdd) {
        logger.debug("-> GuiPlugin: registering palette " + toAdd);

        if (_plugin.isGuiPresent()) {
            Component c = toAdd.getComponent();

            if (c instanceof JToolBar bar) {
                Component[] components = bar.getComponents();
                LinkedHashSet<AbstractButton> set = new LinkedHashSet<>();
                for (Component comp : components) {
                    set.add((AbstractButton) comp);
                }
                Toolbar toolbar = _plugin.getGui().getToolbar();
                toolbar.addTools(set, bar.getName());
            } else {

                // Probably never comes up, but just in case we don't have a menubar, we add the component directly.
                JPanel panel = new JPanel();
                panel.add(c);
                _workbench.addStaticPanel(panel);
            }
        }
    }

    /*
     * @see de.renew.gui.PaletteHolder#removePalette(CH.ifa.draw.util.Palette)
     */
    @Override
    public void removePalette(Palette toRemove) {
        logger.debug("-> GuiPlugin: removing palette " + toRemove);
        _palettes.remove(toRemove);

        if (_plugin.isGuiPresent()) {
            Component c = toRemove.getComponent();
            if (c instanceof JToolBar bar) {
                Toolbar toolbar = _plugin.getGui().getToolbar();
                toolbar.removeTools(bar.getName());
            }
        }

    }

    /*
     * @see de.renew.gui.PaletteHolder#createToolButton(javax.swing.Icon, javax.swing.Icon, java.lang.String, CH.ifa.draw.framework.Tool)
     */
    @Override
    public ToolButton createToolButton(Icon icon, Icon selectedIcon, String toolName, Tool tool) {
        logger.debug("the delegator is " + _eventDelegator);
        return new ToolButton(_eventDelegator, icon, selectedIcon, toolName, tool);
    }

    /*
     * @see de.renew.gui.PaletteHolder#createToolButton(java.lang.String, java.lang.String, CH.ifa.draw.framework.Tool)
     */
    @Override
    public ToolButton createToolButton(String path, String toolName, Tool tool) {
        return new ToolButton(_eventDelegator, path, toolName, tool);
    }

    /**
     * The class that delegates the PaletteListener events to the
     * presently open Gui, if any
     * @author Joern Schumacher
     */
    private static class GuiEventDelegator implements PaletteListener {
        @Override
        public void paletteUserOver(ToolButton button, boolean inside) {
            CPNApplication gui = GuiPlugin.getCurrent().getGui();
            if (gui != null) {
                gui.paletteUserOver(button, inside);
            }
        }

        @Override
        public void paletteUserSelected(ToolButton button, boolean inside) {
            CPNApplication gui = GuiPlugin.getCurrent().getGui();
            if (gui != null) {
                gui.paletteUserSelected(button, inside);
            }
        }
    }

    @Override
    public TextFigureToolButtonRegistry getTextFigureToolButtonRegistry() {
        return this.textFigureToolButtonRegistry;
    }
}