package de.renew.export.io.exportcli;

import java.io.File;
import java.util.Arrays;
import java.util.List;

import CH.ifa.draw.framework.Drawing;
import CH.ifa.draw.io.DrawingFileHelper;
import CH.ifa.draw.io.exportFormats.ExportFormat;
import CH.ifa.draw.util.GUIProperties;
import de.renew.export.io.exportFormats.EPSExportFormat;
import de.renew.export.io.exportFormats.PDFExportFormat;
import de.renew.export.io.exportFormats.PNGExportFormat;
import de.renew.export.io.exportFormats.SVGExportFormat;
import de.renew.plugin.PluginManager;
import de.renew.plugin.PluginProperties;

/**
 * Reads and writes content to/from file by given filename.
 * Automatically finds file type and creates fitting format.
 */
public class Export {
    /**
     * List of ExportFormats.
     */
    public static List<ExportFormat> _exportFormats = Arrays
        .asList((ExportFormat) new PDFExportFormat(), new PNGExportFormat(), new EPSExportFormat());

    /**
     * Main method.
     *
     * @param args Array containing inFileName on position [0] and outFileName on position [1]
     * @throws Exception if export fails
     */
    public static void main(String[] args) throws Exception {
        //        // minimal Renew logging configuration that forwards messages to the Console
        //        ConsoleAppender appender = new ConsoleAppender();
        //        Logger logger = Logger.getLogger("CH.ifa.draw");
        //        logger.addAppender(appender);
        //        logger.setLevel(Level.INFO);
        //        logger = Logger.getLogger("de.renew");
        //        logger.addAppender(appender);
        //        logger.setLevel(Level.INFO);
        PluginManager.configureLogging();
        GUIProperties.setProperties(PluginProperties.getUserProperties());

        if (args.length >= 2) {
            String inFileName = args[0];
            String outFileName = args[1];

            File inFile = new File(inFileName);
            Drawing netDrawing = DrawingFileHelper.loadDrawing(inFile, System.out::println);

            ExportFormat exportFormat = guessExportFormat(outFileName);
            if (exportFormat == null) {
                exportFormat = new PDFExportFormat();
                outFileName += ".pdf";
            }
            File outFile = new File(outFileName);
            // check if diretories exist
            File dir = outFile.getParentFile();
            if (dir != null && !dir.exists()) {
                System.out.println("Creating directory " + dir + " because it does not exist.");
                outFile.getParentFile().mkdirs();
            }
            System.out.println(
                "Exporting " + inFile.getAbsolutePath() + " to " + outFile.getAbsolutePath() + ".");
            exportFormat.export(netDrawing, outFile);
        }
    }

    private static ExportFormat guessExportFormat(String outFileName) {
        String extension = getFileExtension(outFileName);
        ExportFormat format = null;
        switch (extension.toLowerCase()) {
            case "pdf":
                format = new PDFExportFormat();
                break;
            case "eps":
                format = new EPSExportFormat();
                break;
            case "png":
                format = new PNGExportFormat();
                break;
            case "svg":
                format = new SVGExportFormat();
                break;
        }
        return format;
    }

    private static String getFileExtension(String fileName) {
        String extension = "";
        int i = fileName.lastIndexOf('.');
        if (i > 0) {
            extension = fileName.substring(i + 1);
        }
        return extension;
    }
}
