/*
 * @(#)ToolButton.java 5.1
 *
 */
package CH.ifa.draw.standard;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import javax.swing.*;

import CH.ifa.draw.DrawPlugin;
import CH.ifa.draw.application.DrawApplication;
import CH.ifa.draw.framework.Tool;
import CH.ifa.draw.util.PaletteListener;
import de.renew.plugin.PluginManager;
import de.renew.plugin.PluginProperties;


/**
 * A PaletteButton that is associated with a tool.
 *
 * @see Tool
 */
public class ToolButton implements ActionListener, MouseListener, KeyListener {
    private static final String PROP_PRE = "de.renew.keymap.";
    public static final int POPUP_MENU = 0;
    public static final int DIALOG = 1;

    public static org.apache.log4j.Logger logger =
        org.apache.log4j.Logger.getLogger(ToolButton.class);
    private String fName;
    private Tool fTool;
    private JToggleButton fButton;
    private PaletteListener fListener;
    private char keyChar;
    private ArrayList<Component> additionalComponents = new ArrayList<>();
    private int alternativeMode = DIALOG;

    public ToolButton(PaletteListener listener, String iconName, String name, Tool tool) {
        ImageIcon[] icons = createIconsByGuess(iconName);
        createButton(listener, icons[0], icons[1], name, tool);
        associateKey(name.replaceAll(" ", "_"));
    }

    public ToolButton(
        PaletteListener listener, Icon icon, Icon selectedIcon, String name, Tool tool)
    {
        createButton(listener, icon, selectedIcon, name, tool);
        associateKey(name.replaceAll(" ", "_"));
    }

    private void associateKey(String modifiedName) {
        DrawPlugin app = DrawPlugin.getCurrent();
        if (app != null) {
            PluginProperties props = app.getProperties();
            if (props.getBoolProperty(PROP_PRE + "use-mapping")) {
                String key = props.getProperty(PROP_PRE + modifiedName);
                if (key != null) {
                    if (logger.isDebugEnabled()) {
                        logger.debug(
                            ToolButton.class.getName() + ": found key mapping entry: " + key);
                    }
                    if (key.trim().length() == 1) {
                        //assumption: if a tool button is created the gui exists

                        final ToolButton handle = this;
                        Runnable r = new Runnable() {
                            int count = 0;

                            @Override
                            public void run() {
                                DrawApplication gui = DrawPlugin.getGui();

                                if (gui != null) {

                                    JFrame frame = gui.getFrame();
                                    if (frame != null) {
                                        frame.addKeyListener(handle);
                                    } else if (count < 10) {
                                        count++;
                                        EventQueue.invokeLater(this);
                                    }
                                } else if (count < 10) {
                                    count++;
                                    EventQueue.invokeLater(this);
                                }


                            }
                        };
                        EventQueue.invokeLater(r);
                        keyChar = key.trim().charAt(0);
                    }
                }
            }
        }
    }

    private void createButton(
        PaletteListener listener, Icon icon, Icon selectedIcon, String name, Tool tool)
    {

        fButton = new JToggleButton();

        logger.debug("ToolButton overwrites setSelected method in it's model.");
        fButton.setModel(new ToolButtonModel());
        fButton.setIcon(icon);
        fButton.setToolTipText(name);
        fButton.setName(name);
        if (selectedIcon != null) {
            fButton.setPressedIcon(selectedIcon);
        }
        //fButton.addActionListener(this);
        fButton.addMouseListener(this);
        fTool = tool;
        fName = name;
        fListener = listener;
    }

    public Tool tool() {
        return fTool;
    }

    public void setTool(Tool tool) {
        fTool = tool;
    }

    public String name() {
        return fName;
    }

    public String getHotkey() {
        if (Character.isLetterOrDigit(keyChar) || keyChar == '.') {
            return " (" + keyChar + ")";
        }
        return "";
    }

    /**
     * Adds an additional component to be presented in an alternative view mode.
     *
     * @param component
     */
    public void addAdditionalComponent(Component component) {
        additionalComponents.add(component);
    }

    /**
     * Remove an additional component if it exists.
     *
     * @param component
     */
    public void removeAdditionalComponent(Component component) {
        additionalComponents.remove(component);
    }

    /**
     * Set the style in which additional components are presented.
     * Currently supported modes:
     * POPUP_MENU: Only considers components from type {@link JMenuItem}.
     * DIALOG: Standard. Puts each component in a {@link JTabbedPane} and presents them in a dialog.
     *
     * @param mode
     */
    public void setAlternativeMode(int mode) {
        alternativeMode = mode;
    }

    public Object attributeValue() {
        return tool();
    }

    public AbstractButton button() {
        return fButton;
    }

    public void mouseClicked(MouseEvent e) {

        if (SwingUtilities.isRightMouseButton(e) || e.isControlDown()) {
            if (!additionalComponents.isEmpty()) {
                switch (alternativeMode) {
                    case POPUP_MENU:
                        showAdditionalComponentsMenu(e);
                        break;
                    case DIALOG:
                        showAdditionalComponentsDialog();
                        break;
                }
            }
        } else {
            boolean doubleClick = (e.getClickCount() > 1);
            fListener.paletteUserSelected(this, doubleClick);
        }
    }

    @Override
    public void mousePressed(MouseEvent e) {

    }

    @Override
    public void mouseReleased(MouseEvent e) {}

    @Override
    public void mouseEntered(MouseEvent e) {
        fListener.paletteUserOver(this, true);
    }

    @Override
    public void mouseExited(MouseEvent e) {
        fListener.paletteUserOver(this, false);
    }

    public final void actionPerformed(ActionEvent e) {
        fListener.paletteUserSelected(this, true);
    }

    private void showAdditionalComponentsMenu(MouseEvent e) {
        JPopupMenu menu = new JPopupMenu();
        boolean isEmpty = true;
        for (Component c : additionalComponents) {
            if (c instanceof JMenuItem) {
                menu.add(c);
                isEmpty = false;
            }
        }

        if (!isEmpty) {
            Point p = e.getLocationOnScreen();
            Point p2 = button().getLocationOnScreen();

            menu.show(button(), p.x - p2.x, p.y - p2.y);
        }
    }

    private void showAdditionalComponentsDialog() {

        JTabbedPane tabbedPane = new JTabbedPane();
        for (Component c : additionalComponents) {
            tabbedPane.add(c);
        }

        final JDialog dialog = new JDialog(SwingUtilities.getWindowAncestor(button()));
        dialog.setLocationRelativeTo(button());
        dialog.add(tabbedPane);
        //dialog.setUndecorated(true);

        dialog.addWindowFocusListener(new WindowAdapter() {
            @Override
            public void windowLostFocus(WindowEvent e) {
                dialog.setVisible(false);
                dialog.dispose();
            }
        });

        dialog.pack();
        dialog.validate();
        dialog.setVisible(true);
    }

    private static ImageIcon[] createIconsByGuess(final String name) {
        ImageIcon[] result = new ImageIcon[2];
        ImageIcon firstGuess;
        try {
            firstGuess = createIcon(name + ".gif");
        } catch (Exception e) {
            logger.error("Guess for " + name + ".gif failed with " + e);
            firstGuess = null;
        }
        if (firstGuess != null) {
            result[0] = firstGuess;
            result[1] = null;
        } else {
            result[0] = createIcon(name + "1.gif");
            result[1] = createIcon(name + "2.gif");
        }
        return result;
    }

    private static ImageIcon createIcon(String name) {
        URL iconURL = PluginManager.getInstance().getBottomClassLoader().getResource(name);

        if (iconURL == null) {
            File icon = new File(name);
            if (!icon.isFile()) {
                return null;
            }
            try {
                iconURL = icon.toURI().toURL();
            } catch (MalformedURLException e) {
                return null;
            }
        }
        return new ImageIcon(iconURL);
    }

    /*
     * (non-Javadoc)
     *
     * @see java.awt.event.KeyListener#keyPressed(java.awt.event.KeyEvent)
     */
    public void keyPressed(KeyEvent e) {
        if (!e.isControlDown() && !e.isAltDown() && !e.isMetaDown()) {
            if (e.getKeyChar() == keyChar) {
                fListener.paletteUserSelected(this, false);
            } else if (e.getKeyChar() == Character.toUpperCase(keyChar)) {
                fListener.paletteUserSelected(this, true);
            }
        }
    }

    public void keyReleased(KeyEvent e) {}

    public void keyTyped(KeyEvent e) {}

    public class ToolButtonModel extends JToggleButton.ToggleButtonModel {

        @Override
        public void setSelected(boolean b) {
            // We want to control when the button is actually selected (only when clicked, not dragged)
        }

        public void setTrulySelected(boolean b) {
            super.setSelected(b);
        }
    }
}