/*
 * @(#)DrawingView.java 5.1
 *
 */

package CH.ifa.draw.framework;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.geom.AffineTransform;
import java.awt.image.ImageObserver;
import java.awt.print.Printable;
import java.util.Vector;


/**
 * DrawingView renders a Drawing and listens to its changes.
 * It receives user input and delegates it to the current tool.
 * <hr>
 * <b>Design Patterns</b><P>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld026.htm>Observer</a></b><br>
 * DrawingView observes drawing for changes via the DrawingListener interface.<br>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld032.htm>State</a></b><br>
 * DrawingView plays the role of the StateContext in
 * the State pattern. Tool is the State.<br>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld034.htm>Strategy</a></b><br>
 * DrawingView is the StrategyContext in the Strategy pattern
 * with regard to the UpdateStrategy. <br>
 * DrawingView is the StrategyContext for the PointConstrainer.
 *
 * @see Drawing
 * @see Painter
 * @see Tool
 */
public interface DrawingView extends ImageObserver, DrawingChangeListener, Printable {

    /**
     * Sets the view's editor.
     * @param editor editor to be set
     */
    void setEditor(DrawingEditor editor);

    /**
     * Gets the current tool.
     * @return current tool
     */
    Tool tool();

    /**
     * Gets the drawing.
     * @return current drawing
     */
    Drawing drawing();

    /**
     * Sets and installs another drawing in the view.
     * @param d drawing to be set/installed
     */
    void setDrawing(Drawing d);

    /**
     * Gets the editor.
     * @return editor
     */
    DrawingEditor editor();

    /**
     * Adds a figure to the drawing.
     * @param figure figure to added
     * @return the added figure.
     */
    Figure add(Figure figure);

    /**
     * Removes a figure from the drawing.
     * @param figure vector of figures to be removed
     * @return the removed figure
     */
    Figure remove(Figure figure);

    /**
     * Adds a vector of figures to the drawing.
     * @param figures vector of figures to be removed
     */
    void addAll(Vector<Figure> figures);

    /**
     * Removes a vector of figures from the drawing.
     * @param figures vector of figures to be removed
     */
    void removeAll(Vector<Figure> figures);

    /**
     * Gets the size of the drawing.
     * @return size of this drawing
     */
    Dimension getSize();

    /**
     * Gets the minimum dimension of the drawing.
     * @return minimum size of this drawing
     */
    Dimension getMinimumSize();

    /**
     * Gets the preferred dimension of the drawing.
     * @return preferred dimension of this drawing
     */
    Dimension getPreferredSize();

    /**
     * Sets the current display update strategy.
     * @param updateStrategy update Strategy to be set
     */
    void setDisplayUpdate(Painter updateStrategy);

    /**
     * Sorts the given vector of figures into Z-order.
     * @param figures figures included in given vector
     * @return a vector with the ordered figures.
     */
    Vector<Figure> inZOrder(Vector<Figure> figures);

    /**
     * Gets the currently selected figures.
     * @return a vector with the selected figures. The vector
     * is a copy of the current selection.
     */
    Vector<Figure> selection();

    /**
     * Gets an enumeration over the currently selected figures.
     * @return enumeration of selected figures
     */
    FigureEnumeration selectionElements();

    /**
     * Gets the currently selected figures in Z order.
     * @see #selection
     * @return a vector with the selected figures. The vector
     * is a copy of the current selection.
     */
    Vector<Figure> selectionZOrdered();

    /**
     * Gets the number of selected figures.
     * @return number of selected figures
     */
    int selectionCount();

    /**
     * Adds a figure to the current selection.
     * @param figure figure to be added into selection
     */
    void addToSelection(Figure figure);

    /**
     * Adds an enumeration of figures to the current selection.
     * @param figures figures to be added
     */
    void addToSelectionAll(FigureEnumeration figures);

    /**
     * Adds a vector of figures to the current selection.
     * @param figures figures included in vector to be added
     */
    void addToSelectionAll(Vector<Figure> figures);

    /**
     * Removes a figure from the selection.
     * @param figure figure to be removed from selection
     */
    void removeFromSelection(Figure figure);

    /**
     * Removes an enumeration of figures from the current selection.
     * @param figures figures to be removed
     */
    void removeFromSelectionAll(FigureEnumeration figures);

    /**
     * Removes a vector of figures from the current selection.
     * @param figures figures included in vector to be removed
     */
    void removeFromSelectionAll(Vector<Figure> figures);

    /**
     * If a figure isn't selected it is added to the selection.
     * Otherwise, it is removed from the selection.
     * @param figure figure added if selected, otherwise removed
     */
    void toggleSelection(Figure figure);

    /**
     * Clears the current selection.
     */
    void clearSelection();

    /**
     * Toggles a vector of figures.
     * If a figure isn't selected it is added to the selection.
     * Otherwise, it is removed from the selection.
     * @param figures figures vector added if selected, otherwise removed
     */
    void toggleSelectionAll(Vector<Figure> figures);

    /**
     * Toggles an enumeration of figures.
     * If a figure isn't selected it is added to the selection.
     * Otherwise, it is removed from the selection.
     * @param figures figures to be toggled
     */
    void toggleSelectionAll(FigureEnumeration figures);

    /**
     * Invalidates the handles of the current selection.
     * This means that the cached set of handles will be
     * re-calculated next time the selection's handles are
     * queried.
     */
    void selectionInvalidateHandles();

    /**
     * Gets the current selection as a FigureSelection. A FigureSelection
     * can be cut, copied, pasted.
     * @return current selection
     */
    FigureSelection getFigureSelection();

    /**
     * Finds a handle at the given coordinates.
     * @param x point on x-axis
     * @param y point on y-axis
     * @return the hit handle, null if no handle is found.
     */
    Handle findHandle(int x, int y);

    /**
     * Gets the position of the last click inside the view.
     * @return position of last click inside view
     */
    Point lastClick();

    /**
     * Gets the position of the last mouse event inside the view.
     * @return current position of mouse inside view
     */
    Point getCurrentMousePosition();

    /**
     * Sets the current point constrainer.
     * @param p point to be set
     */
    void setConstrainer(PointConstrainer p);

    /**
     * Gets the current constrainer.
     * @return the currently active PointConstrainer, null if none is set
     */
    PointConstrainer getConstrainer();

    /**
     * Checks whether the drawing has some accumulated damage
     * and informs all views about the required update,
     * if neccessary.
     */
    void checkDamage();

    /**
     * Repair the damaged area of this view.
     */
    void repairDamage();

    /**
     * Paints the drawing view. The actual drawing is delegated to
     * the current update strategy.
     * @param g graphic to be painted in
     * {@link Painter}
     */
    void paint(Graphics g);

    /**
     * Creates an image with the given dimensions.
     * @param width width of image
     * @param height height of image
     * @return created image with given parameters
     */
    Image createImage(int width, int height);

    /**
     * Gets a graphic to draw into.
     * @return graphic
     */
    Graphics getGraphics();

    /**
     * Gets the background color of the DrawingView.
     * @return background color
     */
    Color getBackground();

    /**
     * Sets the background color of the DrawingView.
     * @param c new background color
     */
    void setBackground(Color c);

    /**
     * Draws the contents of the drawing view.
     * The view has three layers: background, drawing, handles.
     * The layers are drawn in back to front order.
     * @param g graphic to be drawn into
     */
    void drawAll(Graphics g);

    /**
     * Draws the currently active handles.
     * @param g graphic to be drawn into
     */
    void drawHandles(Graphics g);

    /**
     * Draws the drawing.
     * @param g graphic to be drawn into
     */
    void drawDrawing(Graphics g);

    /**
     * Draws the background. If a background pattern is set it
     * is used to fill the background. Otherwise, the background
     * is filled in the background color.
     * @param g graphic to be drawn into
     */
    void drawBackground(Graphics g);

    /**
     * Sets the cursor of the DrawingView.
     * @param c cursor to be set
     */
    void setCursor(Cursor c);

    /**
     * Freezes the view by acquiring the drawing lock.
     * @see Drawing#lock
     */
    void freezeView();

    /**
     * Unfreezes the view by releasing the drawing lock.
     * @see Drawing#unlock
     */
    void unfreezeView();

    /**
     * Moves Selection by given parameters.
     * @param dx x amount on x-axis
     * @param dy y amount on y-axis
     */
    void moveSelection(int dx, int dy);

    /**
     * Moves the view of a ScrollPane, so the given figure
     * will be visible.
     * @param fig figure to be made visible
     */
    void showElement(Figure fig);

    /**
     * Get the transformation of this view's graphics.
     *
     * @return The AffineTransform object affecting this views graphics
     */
    AffineTransform getAffineTransform();
}