/*
 * Created on Apr 18, 2003
 */

package CH.ifa.draw.framework;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;

import CH.ifa.draw.io.IFAFileFilter;
import CH.ifa.draw.io.SimpleFileFilter;
import de.renew.plugin.PluginManager;


/**
 * This class creates and manages the drawing type manager.
 * @author Lawrence Cabac
 */
public class DrawingTypeManager {
    /**
     * Logger instance for logging concerning the DrawingTypeManager.
     */
    public static final org.apache.log4j.Logger LOGGER =
        org.apache.log4j.Logger.getLogger(DrawingTypeManager.class);
    private static DrawingTypeManager _dtm;
    private final Hashtable<String, SimpleFileFilter> _drawingTypes;
    private SimpleFileFilter _defaultFileFilter;
    private final Set<DrawingTypeManagerListener> _listeners;

    private DrawingTypeManager() {
        _drawingTypes = new Hashtable<>();
        _listeners = new HashSet<>();
    }

    /**
     * Creates an instance of DrawingTypeManager on the first call of the method.
     * @return created instance
     */
    public static DrawingTypeManager getInstance() {
        //this should be made more save fix!
        //i.e. the existence of the Class should be checked 
        //and maybe whether this implements Diagram (possible?)
        if (_dtm == null) {
            _dtm = new DrawingTypeManager();
            IFAFileFilter ifa = new IFAFileFilter();
            _dtm.register("CH.ifa.draw.standard.StandardDrawing", ifa);
            _dtm.setDefaultFileFilter(ifa);
            return _dtm;
        }
        return _dtm;
    }

    /**
     * Registers a new drawing type.
     * @param drawingName name of the drawing type
     * @param ff matching file filter for type
     * @return previous ff, if there was one before
     */
    public Object register(String drawingName, SimpleFileFilter ff) {
        LOGGER.debug("DrawingTypeManager: registering " + ff + " for drawing type " + drawingName);
        Object previous = _drawingTypes.put(drawingName, ff);
        for (DrawingTypeManagerListener drawingTypeManagerListener : _listeners) {
            drawingTypeManagerListener.typeRegistered(drawingName, ff);
        }

        return previous;
    }

    /**
     * Instantiates a Drawing based on its name.
     * @param name class name of drawing
     * @return drawing with given name, else null if no drawing can be found
     */
    static public Drawing getDrawingFromName(String name) {
        Drawing d = null;

        //LOGGER.debug("The name of the Drawing dtm "+name);
        try {
            ClassLoader classLoader = PluginManager.getInstance().getBottomClassLoader();
            Constructor<?> constructor = Class.forName(name, true, classLoader).getConstructor();
            d = (Drawing) constructor.newInstance();
        } catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
            LOGGER.error(e.getMessage(), e);
        } catch (InvocationTargetException | NoSuchMethodException e) {
            throw new RuntimeException(e);
        }
        return d;
    }

    /**
     * Gets the Drawing associated with the given file filter.
     * @param ff given file filter
     * @return drawing, else null if no drawing can be found
     */
    public static Drawing getDrawingForFilter(SimpleFileFilter ff) {
        Hashtable<String, SimpleFileFilter> hash = getInstance().getDrawingTypes();
        Enumeration<String> enumeration = hash.keys();
        while (enumeration.hasMoreElements()) {
            String key = enumeration.nextElement();
            SimpleFileFilter fileFilter = hash.get(key);
            if (fileFilter.equals(ff)) {
                return getDrawingFromName(key);
            }
        }
        return null;
    }

    /**
     * Returns the mapping of drawing type names to their associated file filters.
     * @return drawing types
     */
    public Hashtable<String, SimpleFileFilter> getDrawingTypes() {
        return _drawingTypes;
    }

    /**
     * Checks whether the name of a DrawingType is included.
     * @param name name to be checked
     * @return boolean, {@code true} if key is contained
     */
    public boolean contains(String name) {
        return _drawingTypes.containsKey(name);
    }

    /**
     * Gets the registered FileFilter for the DrawingType.
     * @param name name of drawing type
     * @return filter
     */
    public SimpleFileFilter getFilter(String name) {
        return _drawingTypes.get(name);

    }

    /**
     * Gets the default Filter.
     * @return default Filter
     */
    public SimpleFileFilter getDefaultFileFilter() {
        return _defaultFileFilter;
    }

    /**
     * Sets the default Filter.
     * @param filter filter to be set
     * @throws IllegalArgumentException the given file filter is unknown
     */
    public void setDefaultFileFilter(SimpleFileFilter filter) {
        if (!_drawingTypes.contains(filter)) {
            throw new IllegalArgumentException(
                "Cannot choose unknown drawing type as default: " + filter);
        }
        LOGGER.debug("DrawingTypeManager: choosing " + filter + " as default drawing type.");
        _defaultFileFilter = filter;
        for (DrawingTypeManagerListener drawingTypeManagerListener : _listeners) {
            drawingTypeManagerListener.defaultTypeChanged(filter);
        }
    }

    /**
     * Adds listener to the DrawingTypeManager.
     * @param listener listener to be added
     */
    public void addListener(DrawingTypeManagerListener listener) {
        _listeners.add(listener);
    }

    /**
     * Removes listener from the DrawingTypeManager.
     * @param listener listener to remove
     */
    public void removeListener(DrawingTypeManagerListener listener) {
        _listeners.remove(listener);
    }

    /**
     * Retracts a drawing type from the set of known types.
     * @param drawingName the name of the retracted drawing type
     */
    public void unregister(String drawingName) {
        LOGGER.debug("DrawingTypeManager: unregistering " + " for drawing type " + drawingName);
        _drawingTypes.remove(drawingName);
        if (drawingName.equals(_defaultFileFilter.getDescription())) {
            Enumeration<String> enu = _drawingTypes.keys();
            if (enu.hasMoreElements()) {
                SimpleFileFilter ff = _drawingTypes.get(enu.nextElement());
                setDefaultFileFilter(ff);
            } else {
                _defaultFileFilter = null;
            }
        }

        for (DrawingTypeManagerListener drawingTypeManagerListener : _listeners) {
            drawingTypeManagerListener.defaultTypeChanged(_defaultFileFilter);
        }
    }
}